/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.framework.spring.config.auth.mock;

import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.context.annotation.Conditional;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.annotation.Order;
import org.springframework.security.config.annotation.authentication.builders.AuthenticationManagerBuilder;
import org.springframework.security.config.annotation.authentication.configurers.provisioning.InMemoryUserDetailsManagerConfigurer;
import org.springframework.security.config.annotation.web.WebSecurityConfigurer;
import org.springframework.security.config.annotation.web.builders.HttpSecurity;
import org.springframework.security.config.annotation.web.configuration.EnableWebSecurity;
import org.springframework.security.config.annotation.web.configuration.WebSecurityConfigurerAdapter;
import org.springframework.security.crypto.factory.PasswordEncoderFactories;
import org.springframework.security.crypto.password.PasswordEncoder;

import com.fasterxml.jackson.annotation.JsonInclude.Include;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.sap.cds.framework.spring.config.auth.HttpSecurityConfigurer;
import com.sap.cds.framework.spring.config.auth.identity.IdentitySecurityConfig;
import com.sap.cds.framework.spring.config.auth.xsuaa.XsuaaSecurityConfig;
import com.sap.cds.services.environment.CdsProperties.Security.Mock.User;
import com.sap.cds.services.runtime.CdsRuntime;

/**
 * The default spring security configuration based on mock users. Active if and only if mock users are configured in the active profile.
 * Mock users are authenticated via basic authentication. By default, passwords are empty.
 */
@Configuration
@ConditionalOnClass(WebSecurityConfigurer.class) // only if spring-security library has been loaded
@AutoConfigureAfter({XsuaaSecurityConfig.class, IdentitySecurityConfig.class}) // Xsuaa/IdentitySecurityConfig have higher priority and should be checked first
@ConditionalOnMissingBean({XsuaaSecurityConfig.class, IdentitySecurityConfig.class}) // don't activate mock users in case of productive config
@Conditional(MockUsersConfiguredCondition.class) // only load in case mock users are configured
@EnableWebSecurity
@Order(1000)
public class MockUsersSecurityConfig extends WebSecurityConfigurerAdapter {

	private final static Logger logger = LoggerFactory.getLogger(MockUsersSecurityConfig.class);
	private final static ObjectMapper mapper;

	static {
		mapper = new ObjectMapper();
		mapper.setSerializationInclusion(Include.NON_NULL);
	}

	@Autowired
	private HttpSecurityConfigurer httpSecurityConfigurer;

	@Autowired
	private CdsRuntime runtime;

	@Override
	public void configure(AuthenticationManagerBuilder auth) throws Exception {

		List<User> mockUsers = runtime.getEnvironment().getCdsProperties().getSecurity().getMock().getUsers();
		if (mockUsers.isEmpty()) {
			// go with standard spring security auth manager if there is no mock user configured (i.e. there will be default user 'user' with password written to console)
			logger.warn("No mock users found in configuration. Spring standard authentication manager with default user 'user' (password written to console) is available");
			super.configure(auth);
			return;
		}

		InMemoryUserDetailsManagerConfigurer<AuthenticationManagerBuilder> authConfigurer = auth.inMemoryAuthentication();
		PasswordEncoder encoder = PasswordEncoderFactories.createDelegatingPasswordEncoder();
		authConfigurer.passwordEncoder(encoder);
		for (User user : mockUsers) {
			if (!user.isValid()) {
				logger.warn("Skipping invalid mock user defintion {} ('name' and 'password' required)", stringifyUser(user));
			} else {
				authConfigurer.withUser(user.getName()).password(encoder.encode(user.getPassword())).authorities(user.getRoles().toArray(new String[0]));
				logger.info("Added mock user {}", stringifyUser(user));
			}
		}
	}

	@Override
	public void configure(HttpSecurity http) throws Exception {

		http.csrf().disable();

		httpSecurityConfigurer.configure(http);

		http.httpBasic();

		logger.info("*************************************************************************");
		logger.info("*  Security configuration based on mock users found in active profile.  *");
		logger.info("*                 !!! NEVER USE IN PRODUCTIVE MODE !!!                  *");
		logger.info("*************************************************************************");
	}

	private String stringifyUser(User user) {
		try {
			return mapper.writeValueAsString(user);
		} catch (Exception e) { // NOSONAR
			return user.getName();
		}
	}

}
