/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.framework.spring.config.runtime;

import java.util.List;
import java.util.Locale;
import java.util.function.Function;

import javax.servlet.http.HttpServletRequest;

import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.Order;
import org.springframework.http.HttpHeaders;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.core.userdetails.User;
import org.springframework.security.oauth2.client.OAuth2AuthorizedClient;
import org.springframework.security.oauth2.client.OAuth2AuthorizedClientService;
import org.springframework.security.oauth2.client.authentication.OAuth2AuthenticationToken;
import org.springframework.security.oauth2.server.resource.authentication.JwtAuthenticationToken;
import org.springframework.web.context.request.RequestAttributes;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import com.sap.cds.services.authentication.AuthenticationInfo;
import com.sap.cds.services.authentication.BasicAuthenticationInfo;
import com.sap.cds.services.authentication.JwtTokenAuthenticationInfo;
import com.sap.cds.services.authentication.JwtTokenWithForwardJwtAuthenticationInfo;
import com.sap.cds.services.runtime.AuthenticationInfoProvider;

@Configuration
@ConditionalOnClass(SecurityContextHolder.class)
public class SpringAuthenticationInfoProvider {

	private static final String AUTH_BEARER_PREFIX = "bearer";

	private static final String X_IAS_TOKEN_HEADER = "x-ias-token"; // as used by the AppRouter to send additional ID token

	@Configuration
	@ConditionalOnClass(UsernamePasswordAuthenticationToken.class)
	public static class BasicAuth {

		@Bean
		@Order(Ordered.LOWEST_PRECEDENCE - 2)
		public AuthenticationInfoProvider basicAuthenticationInfoProvider() {
			return new SecurityContextAuthenticationInfoProvider((auth) -> {
				if (auth instanceof UsernamePasswordAuthenticationToken) {
					Object principal = ((UsernamePasswordAuthenticationToken) auth).getPrincipal();
					if (principal instanceof User) {
						User user = (User) principal;
						return new BasicAuthenticationInfo(user.getUsername(), user.getPassword());
					}
				}
				return null;
			});
		}

	}

	@Configuration
	@ConditionalOnClass(OAuth2AuthenticationToken.class)
	public static class OAuth2Client {

		@Bean
		@Order(Ordered.LOWEST_PRECEDENCE - 1)
		public AuthenticationInfoProvider oauth2ClientAuthenticationInfoProvider(List<OAuth2AuthorizedClientService> clientServices) {
			return new SecurityContextAuthenticationInfoProvider((auth) -> {
				if(auth instanceof OAuth2AuthenticationToken) {
					String authorizedClientRegistrationId = ((OAuth2AuthenticationToken) auth).getAuthorizedClientRegistrationId();
					String principalName = ((OAuth2AuthenticationToken) auth).getName();

					for(OAuth2AuthorizedClientService clientService : clientServices) {
						OAuth2AuthorizedClient authorizedClient = clientService.loadAuthorizedClient(authorizedClientRegistrationId, principalName);
						if(authorizedClient != null) {
							return new JwtTokenAuthenticationInfo(authorizedClient.getAccessToken().getTokenValue());
						}
					}
				}
				return null;
			});
		}

	}

	@Configuration
	@ConditionalOnClass(JwtAuthenticationToken.class)
	public static class OAuth2ResourceServer {

		@Bean
		@Order(Ordered.LOWEST_PRECEDENCE)
		public AuthenticationInfoProvider oauth2ResourceServerAuthenticationInfoProvider() {
			return new SecurityContextAuthenticationInfoProvider((auth) -> {
				if (auth instanceof JwtAuthenticationToken) {
					String requestToken = ((JwtAuthenticationToken) auth).getToken().getTokenValue();
					String forwardToken = getForwardToken(requestToken);
					if (forwardToken != null) {
						return new JwtTokenWithForwardJwtAuthenticationInfo(requestToken, forwardToken);
					} else {
						return new JwtTokenAuthenticationInfo(requestToken);
					}
				}
				return null;
			});
		}

		// An IAS-Token might be initiated in the Authorization header or as additional x-ias-token header
		private static String getForwardToken(String requestToken) {
			 
			HttpServletRequest request = null;
			RequestAttributes reqAttrs = RequestContextHolder.getRequestAttributes();
			if (reqAttrs != null && reqAttrs instanceof ServletRequestAttributes) {
				request = ((ServletRequestAttributes)reqAttrs).getRequest();
			} else {
				return null; // not in scope of a web request
			}
				
			String authHeader = request.getHeader(HttpHeaders.AUTHORIZATION);
			if (authHeader != null && authHeader.toLowerCase(Locale.ENGLISH).startsWith(AUTH_BEARER_PREFIX)) {
				String initialToken = authHeader.substring(AUTH_BEARER_PREFIX.length()).trim();
				if (!initialToken.equals(requestToken)) { // there was an exchange
					return initialToken;
				}
			}

			String fwdToken = request.getHeader(X_IAS_TOKEN_HEADER);
			if (fwdToken != null && !fwdToken.isEmpty()) {
				return fwdToken;
			}

			return null;
		}
	}

	private static class SecurityContextAuthenticationInfoProvider implements AuthenticationInfoProvider {

		private final Function<Authentication, AuthenticationInfo> authenticationMapper;
		private AuthenticationInfoProvider previous;

		public SecurityContextAuthenticationInfoProvider(Function<Authentication, AuthenticationInfo> authenticationMapper) {
			this.authenticationMapper = authenticationMapper;
		}

		@Override
		public AuthenticationInfo get() {
			Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
			AuthenticationInfo authenticationInfo = authenticationMapper.apply(authentication);
			return authenticationInfo == null ? previous.get() : authenticationInfo;
		}

		@Override
		public void setPrevious(AuthenticationInfoProvider previous) {
			this.previous = previous;
		}

	}

}
