/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.framework.spring.config.datasource;

import java.util.Arrays;
import java.util.List;

import javax.sql.DataSource;

import org.springframework.beans.factory.FactoryBean;
import org.springframework.boot.context.properties.bind.BindResult;
import org.springframework.boot.context.properties.bind.Bindable;
import org.springframework.boot.context.properties.bind.Binder;
import org.springframework.boot.jdbc.DataSourceBuilder;
import org.springframework.core.env.Environment;

import com.sap.cds.services.datasource.DataSourceDescriptor;
import com.sap.cds.services.utils.datasource.DataSourceUtils;

/**
 * Auto-configures a DataSource in Spring, based on a given {@link DataSourceDescriptor}
 */
public class DataSourceBeanFactory implements FactoryBean<DataSource> {

	private final DataSourceDescriptor descriptor;
	private final Environment environment;
	private DataSource dataSource;

	public DataSourceBeanFactory(DataSourceDescriptor descriptor, Environment environment) {
		this.descriptor = descriptor;
		this.environment = environment;
	}

	@Override
	public DataSource getObject() throws Exception {
		if(dataSource == null) {
			DataSourceBuilder<?> builder = DataSourceBuilder.create();
			builder.driverClassName(descriptor.getDriverClassName());
			builder.url(descriptor.getUrl());
			builder.username(descriptor.getUsername());
			builder.password(descriptor.getPassword());
			DataSource ds = builder.build();

			Binder binder = Binder.get(environment);
			BindResult<DataSource> bindResult;
			Bindable<DataSource> bindableDs = Bindable.ofInstance(ds);
			for (String section: getDataSourceSections(descriptor.getName())) {
				bindResult = binder.bind(section, bindableDs);
				if (bindResult.isBound()) {
					ds = bindResult.get();
					break;
				}
			}
			dataSource = ds;
		}
		return dataSource;
	}

	private static List<String> getDataSourceSections(String name) {
		return Arrays.asList(DataSourceUtils.getDataSourceSection(name, DataSourceUtils.PoolType.HIKARI)
				, DataSourceUtils.getDataSourceSection(name, DataSourceUtils.PoolType.TOMCAT)
				, DataSourceUtils.getDataSourceSection(name, DataSourceUtils.PoolType.DBCP2));
	}

	@Override
	public Class<DataSource> getObjectType() {
		return DataSource.class;
	}

}
