/**************************************************************************
 * (C) 2019-2023 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.framework.spring.utils;

import java.util.Arrays;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.WebApplicationType;
import org.springframework.boot.autoconfigure.SpringBootApplication;
import org.springframework.boot.builder.SpringApplicationBuilder;
import org.springframework.context.annotation.ComponentScan;

import com.sap.cds.services.mt.DeploymentService;
import com.sap.cds.services.mt.TenantProviderService;
import com.sap.cds.services.runtime.CdsRuntime;

/**
 * Class providing a main method to upgrade the tenant.
 * This main method starts the whole server. Therefore, all background tasks that might be registered will also start.
 * If this could lead to problems, they need to be deactivated.
 */
@SpringBootApplication
@ComponentScan(basePackages = {"${cds.multitenancy.deploy.component-scan:acme.xyy}", "${cds.deploy.component-scan:acme.xyz}"})
public class Deploy {

	private static final Logger logger = LoggerFactory.getLogger(Deploy.class);

	public static void main(String[] args) throws Exception {
		try {
			// deactivate authentication and messaging
			System.setProperty("cds.security.xsuaa.enabled", "false");
			System.setProperty("cds.security.mock.enabled", "false");
			System.setProperty("cds.messaging.webhooks.enabled", "false");

			// start without web server
			CdsRuntime runtime = new SpringApplicationBuilder(Deploy.class).web(WebApplicationType.NONE).run(new String[0]).getBean(CdsRuntime.class);
			DeploymentService deploymentService = runtime.getServiceCatalog().getService(DeploymentService.class, DeploymentService.DEFAULT_NAME);
			TenantProviderService tenantService = runtime.getServiceCatalog().getService(TenantProviderService.class, TenantProviderService.DEFAULT_NAME);
			// make sure the property names above stay correct
			assert !runtime.getEnvironment().getCdsProperties().getSecurity().getXsuaa().isEnabled();
			assert !runtime.getEnvironment().getCdsProperties().getSecurity().getMock().isEnabled();
			assert !runtime.getEnvironment().getCdsProperties().getMessaging().getWebhooks().isEnabled();

			if (deploymentService == null) {
				logger.error("Failed to find DeploymentService");
				logErrorMessage();
				System.exit(2);
			}

			String[] tenants;
			if (args.length == 0) {
				logger.info("Obtaining all subscribed tenants");
				tenants = tenantService.readTenants().toArray(new String[0]);
			} else {
				tenants = args;
			}

			if (tenants.length != 0) {
				long timestamp = System.currentTimeMillis();
				runtime.requestContext().privilegedUser().run(context -> {
					logger.info("Starting upgrade for tenant(s) '{}'", String.join(",", tenants));
					deploymentService.upgrade(Arrays.asList(tenants), null);
				});
				logger.info("Upgrade finished successfully in {}s", (System.currentTimeMillis() - timestamp) / 1000);
			} else {
				logger.info("No tenants specified and there are no subscribed tenants. No upgrade necessary.");
			}

			logSuccessMessage();
			System.exit(0);
		} catch (Throwable t) {
			logger.error("Failed to perform tenant upgrade", t);
			logErrorMessage();
			System.exit(1);
		}
	}

	private static void logSuccessMessage() {
		logger.info("*************");
		logger.info("*  SUCCESS  *");
		logger.info("*************");
	}

	private static void logErrorMessage() {
		logger.error("***********");
		logger.error("*  ERROR  *");
		logger.error("***********");
	}

}
