package com.sap.cds.framework.spring.config.auth.identity;

import java.lang.reflect.Field;
import java.util.List;
import java.util.stream.Stream;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.SpringApplication;
import org.springframework.boot.env.EnvironmentPostProcessor;
import org.springframework.core.env.ConfigurableEnvironment;

import com.sap.cds.framework.spring.config.runtime.BootstrapCache;
import com.sap.cds.services.environment.ServiceBinding;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.XsuaaUtils;
import com.sap.cds.services.utils.environment.CdsEnvironmentVariableReader;
import com.sap.cloud.security.config.Environments;
import com.sap.cloud.security.config.cf.CFEnvironment;
import com.sap.cloud.security.spring.config.IdentityServiceConfiguration;

public class IdentityEnvironmentPostProcessor implements EnvironmentPostProcessor {

	private final static Logger logger = LoggerFactory.getLogger(IdentityEnvironmentPostProcessor.class);
	private static boolean identityLibraryAvailable;

	static {
		try {
			identityLibraryAvailable = IdentityServiceConfiguration.class.getName() != null;
		} catch (NoClassDefFoundError e) { // NOSONAR
			identityLibraryAvailable = false;
		}
	}

	@Override
	public void postProcessEnvironment(ConfigurableEnvironment environment, SpringApplication application) {
		if(!identityLibraryAvailable) {
			return;
		}
		
		CdsRuntime runtime = BootstrapCache.get(environment).getCdsRuntime();

		XsuaaUtils xsuaaUtils = new XsuaaUtils(runtime);
		if (!xsuaaUtils.identityAuthenticationConfigured()) {
			return;
		}

		List<ServiceBinding> bindings = xsuaaUtils.getIasServiceBindings(); // IAS takes precedence
		if (bindings.isEmpty()) {
			bindings = xsuaaUtils.getXsuaaServiceBindings();
		}
		if (bindings.isEmpty()) {
			logger.info("No service binding with tag 'identity' or 'xsuaa' found");
			return;
		} else if (bindings.size() > 1) {
			logger.warn("Multiple service bindings with tag 'identity' resp. 'xsuaa' found");
		}

		ServiceBinding binding = bindings.get(0);
		logger.info("Using service binding '{}' to configure identity authentication", binding.getName());

		// TODO: Using reflection as dirty workaround here until java-security provides an API to control the Environment
		CFEnvironment env = CFEnvironment.getInstance(section -> new CdsEnvironmentVariableReader(Stream.of(binding)).apply(section), section -> null);
		try {
			Field field = Environments.class.getDeclaredField("currentEnvironment");
			field.setAccessible(true);
			field.set(null, env);
		} catch (Exception e) {
			throw new InternalError("Missing field com.sap.cloud.security.config.Environments.currentEnvironment");
		}
	}

}
