/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.framework.spring.config.adapter;

import javax.servlet.Servlet;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.BeanFactory;
import org.springframework.beans.factory.BeanFactoryAware;
import org.springframework.beans.factory.SmartFactoryBean;

import com.sap.cds.adapter.AdapterFactory;
import com.sap.cds.adapter.ServletAdapterFactory;
import com.sap.cds.framework.spring.config.runtime.BootstrapCache;

/**
 * Factory bean to create an adapter as bean based on a {@link AdapterFactory}
 */
public class AdapterBeanFactory implements SmartFactoryBean<Object>, BeanFactoryAware {

	private final static Logger logger = LoggerFactory.getLogger(AdapterBeanFactory.class);

	private final String factoryName;
	private BeanFactory beanFactory;
	private Object adapter;

	public AdapterBeanFactory(String factoryName) {
		this.factoryName = factoryName;
	}

	@Override
	public Object getObject() throws Exception {
		if(adapter == null) {
			BootstrapCache bootstrapCache = beanFactory.getBean(BootstrapCache.class);
			AdapterFactory factory = bootstrapCache.getAdapterFactories().get(factoryName);
			if(factory instanceof ServletAdapterFactory) {
				ServletAdapterFactory servletFactory = (ServletAdapterFactory) factory;
				Servlet servlet = servletFactory.create();

				String servletName = servlet.getClass().getSimpleName();
				String basePath = servletFactory.getBasePath();

				adapter = new ServletAdapterRestController(servlet, basePath);
				logger.info("Servlet {} mapped to {}", servletName, basePath);
			} else {
				adapter = factory.create();
				logger.info("Adapter {} initialized", adapter.getClass().getSimpleName());
			}
		}

		return adapter;
	}

	@Override
	public void setBeanFactory(BeanFactory beanFactory) throws BeansException {
		this.beanFactory = beanFactory;
	}

	@Override
	public Class<?> getObjectType() {
		return Object.class;
	}

	@Override
	public boolean isEagerInit() {
		return true;
	}

}
