/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.framework.spring.config.runtime;

import java.util.Locale;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.helpers.MessageFormatter;
import org.springframework.boot.autoconfigure.AutoConfiguration;
import org.springframework.context.MessageSource;
import org.springframework.context.annotation.Bean;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.Order;

import com.sap.cds.services.messages.LocalizedMessageProvider;
import com.sap.cds.services.utils.LocaleUtils;
import com.sap.cds.services.utils.StringUtils;

@AutoConfiguration
public class SpringLocalizedMessageProvider {

	private final static Logger logger = LoggerFactory.getLogger(SpringLocalizedMessageProvider.class);

	@Bean
	@Order(Ordered.LOWEST_PRECEDENCE)
	public LocalizedMessageProvider localizedMessageProvider(MessageSource messageSource) {
		return new LocalizedMessageProvider() {

			@Override
			public String get(String code, Object[] args, Locale locale) {

				try {
					String message = messageSource.getMessage(code, args, null, LocaleUtils.getLocaleForBundle(locale));
					if (message != null) { // NOSONAR
						return message;
					} else {
						return MessageFormatter.arrayFormat(code, args).getMessage();
					}

				} catch (Exception e) {  // NOSONAR
					// be tolerant - we don't want to break the request due to wrong message resolution
					logger.warn("Invalid message format '{}' or incompatible arguments {}", code, StringUtils.toStringSafely(args), e);
					return code;
				}

			}

		};
	}

}
