/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.framework.spring.actuator;

import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.AutoConfiguration;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.context.ApplicationContext;
import org.springframework.security.config.annotation.web.WebSecurityConfigurer;

import com.google.common.collect.Lists;
import com.sap.cds.adapter.ServletUrlResourcePaths;
import com.sap.cds.adapter.ServletUrlResourcePaths.UrlResourcePathVisitor;
import com.sap.cds.adapter.UrlResourcePath;
import com.sap.cds.framework.spring.config.auth.identity.IdentitySecurityConfig;
import com.sap.cds.framework.spring.config.auth.mock.MockUsersSecurityConfig;
import com.sap.cds.framework.spring.config.auth.xsuaa.XsuaaSecurityConfig;
import com.sap.cds.services.environment.CdsProperties.Security;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.info.CdsInfo;

@AutoConfiguration
@ConditionalOnClass(WebSecurityConfigurer.class) // only if spring-security library has been loaded
public class CdsSecurityInfo {

	enum Authentication {
		CUSTOM ("Custom", "<unknown>", "Make sure CDS adapter endpoints are secured accordingly", null),
		MOCK ("Mock users", "Basic", "Not appropriate for productive usage!", MockUsersSecurityConfig.class),
		XSUAA ("XSUAA", "OAuth2", null, XsuaaSecurityConfig.class),
		IDENTITY ("Identity", "OAuth2", null, IdentitySecurityConfig.class);

		private final String name;
		private final String type;
		private final String remark;
		private final Class<?> config;

		private Authentication(String name, String type, String remark, Class<?> config) {
			this.name = name;
			this.type = type;
			this.remark = remark;
			this.config = config;
		}
	}

	@Autowired
	private ApplicationContext applicationContext;

	@Autowired
	private CdsRuntime cdsRuntime;

	public CdsSecurityInfo() {
		CdsSecurityInfoImpl.securityInfo = this;
	}

	private Authentication determineAuthentication() {
		return Stream.of(Authentication.values())
				.filter(e -> e.config != null && applicationContext.getBeanNamesForType(e.config).length > 0)
				.findFirst().orElse(Authentication.CUSTOM);
	}

	@SuppressWarnings("serial")
	private Map<String, Object> authentication() {
		Authentication authentication = determineAuthentication();
		return new LinkedHashMap<String, Object>() {{
			put("name", authentication.name);
			if (authentication.type != null) {
				put("type", authentication.type);
			}
			if (authentication.remark != null) {
				put("remark", authentication.remark);
			}
			Security security = cdsRuntime.getEnvironment().getCdsProperties().getSecurity();
			put("authenticate-unknown-endpoints", security.getAuthentication().isAuthenticateUnknownEndpoints());
			put("authenticate-metadata-endpoints", security.getAuthentication().isAuthenticateMetadataEndpoints());
			put("mode", security.getAuthentication().getMode());
		}};
	}

	@SuppressWarnings("serial")
	private Map<String, Object> authorization() {
		return new LinkedHashMap<String, Object>() {{
			Security security = cdsRuntime.getEnvironment().getCdsProperties().getSecurity();
			put("instance-based-authorization", security.getAuthorization().getInstanceBased().isEnabled());
			put("draft-protection", security.getAuthorization().getDraftProtection().isEnabled());
		}};
	}

	private Map<String, List<String>> publicEndpoints() {
		Authentication authentication = determineAuthentication();
		Map<String, List<String>> result = new LinkedHashMap<>();
		if (authentication != Authentication.CUSTOM) {
			new ServletUrlResourcePaths(cdsRuntime).visit(new UrlResourcePathVisitor() {

				@Override
				public void foundPublicPath(UrlResourcePath publicPath) {
					result.put(publicPath.getPath(), Lists.newArrayList("*"));
				}

				@Override
				public void foundPublicEvents(UrlResourcePath path, Stream<String> publicEvents) {
					List<String> publicEventList = publicEvents.collect(Collectors.toList());
					if (!publicEventList.isEmpty()) {
						result.put(path.getPath(), publicEventList);
					}
				}

			});
		}

		return result;
	}

	public static class CdsSecurityInfoImpl implements CdsInfo {

		private static CdsSecurityInfo securityInfo;

		@SuppressWarnings("serial")
		@Override
		public Map<String, Object> info(Details details) {
			return new LinkedHashMap<String, Object>() {{
				if (securityInfo != null) {
					put("authentication", securityInfo.authentication());
					put("authorization", securityInfo.authorization());
					if (details == Details.HIGH) {
						Map<String, List<String>> publicEndpoints = securityInfo.publicEndpoints();
						if (!publicEndpoints.isEmpty()) {
							put("public CDS adapter endpoints", publicEndpoints);
						}
					}
				} else {
					// can't make use of Authentication.NONE which is not loadable
					put("authentication", new LinkedHashMap<String, Object>() {{
						put("name", "Not configured");
						put("remark", "All endpoints are public!");
					}});
				}
			}};
		}

		@Override
		public String name() {
			return "security";
		}
	}

}
