package com.sap.cds.framework.spring.config.auth.identity;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.SpringApplication;
import org.springframework.boot.env.EnvironmentPostProcessor;
import org.springframework.core.env.ConfigurableEnvironment;

import com.sap.cds.framework.spring.config.runtime.BootstrapCache;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.XsuaaUtils;
import com.sap.cloud.environment.servicebinding.api.ServiceBinding;
import com.sap.cloud.security.config.Environments;
import com.sap.cloud.security.config.ServiceBindingEnvironment;
import com.sap.cloud.security.spring.config.IdentityServiceConfiguration;

public class IdentityEnvironmentPostProcessor implements EnvironmentPostProcessor {

	private final static Logger logger = LoggerFactory.getLogger(IdentityEnvironmentPostProcessor.class);
	private static boolean identityLibraryAvailable;

	static {
		try {
			identityLibraryAvailable = IdentityServiceConfiguration.class.getName() != null;
		} catch (NoClassDefFoundError e) { // NOSONAR
			identityLibraryAvailable = false;
		}
	}

	@Override
	public void postProcessEnvironment(ConfigurableEnvironment environment, SpringApplication application) {
		if(!identityLibraryAvailable) {
			return;
		}

		CdsRuntime runtime = BootstrapCache.get(environment).getCdsRuntime();

		XsuaaUtils xsuaaUtils = new XsuaaUtils(runtime);
		if (!xsuaaUtils.identityAuthenticationConfigured()) {
			return;
		}

		List<ServiceBinding> identityBindings = xsuaaUtils.getIasServiceBindings(); // IAS takes precedence
		if (identityBindings.isEmpty()) {
			logger.info("No service binding with tag 'identity' found");
			return;
		} else if (identityBindings.size() > 1) {
			logger.warn("Multiple service bindings with tag 'identity' found");
		}

		List<ServiceBinding> xsuaaBindings = xsuaaUtils.getXsuaaServiceBindings();
		if (xsuaaBindings.size() > 1) {
			logger.warn("Multiple service bindings with tag 'xsuaa' found");
		}

		List<ServiceBinding> bindings = new ArrayList<>();
		bindings.add(identityBindings.get(0));

		if(!xsuaaBindings.isEmpty()) {
			bindings.add(xsuaaBindings.get(0));
		}

		bindings.forEach(b -> logger.info("Using service binding '{}' to configure identity authentication", b.getName().get()));

		// TODO: Using reflection as dirty workaround here until java-security provides an API to control the Environment
		ServiceBindingEnvironment env = new ServiceBindingEnvironment(() -> bindings);
		try {
			Field field = Environments.class.getDeclaredField("currentEnvironment");
			field.setAccessible(true);
			field.set(null, env);
		} catch (Exception e) {
			throw new InternalError("Missing field com.sap.cloud.security.config.Environments.currentEnvironment", e);
		}
	}

}
