/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.framework.spring.config.auth.mock;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.AutoConfiguration;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnWebApplication;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Conditional;
import org.springframework.core.annotation.Order;
import org.springframework.security.authentication.dao.DaoAuthenticationProvider;
import org.springframework.security.config.annotation.web.builders.HttpSecurity;
import org.springframework.security.config.annotation.web.configuration.EnableWebSecurity;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.security.core.userdetails.UserDetailsService;
import org.springframework.security.provisioning.InMemoryUserDetailsManager;
import org.springframework.security.web.SecurityFilterChain;

import com.fasterxml.jackson.annotation.JsonInclude.Include;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.sap.cds.framework.spring.config.auth.HttpSecurityConfigurer;
import com.sap.cds.framework.spring.config.auth.identity.IdentitySecurityConfig;
import com.sap.cds.framework.spring.config.auth.xsuaa.XsuaaSecurityConfig;
import com.sap.cds.services.environment.CdsProperties.Security.Mock.User;
import com.sap.cds.services.runtime.CdsRuntime;

/**
 * The default spring security configuration based on mock users. Active if and only if mock users are configured in the active profile.
 * Mock users are authenticated via basic authentication. By default, passwords are empty.
 */
@AutoConfiguration(after = {XsuaaSecurityConfig.class, IdentitySecurityConfig.class}) // Xsuaa/IdentitySecurityConfig have higher priority and should be checked first
@ConditionalOnClass(HttpSecurity.class) // only if spring-security library has been loaded
@ConditionalOnMissingBean({XsuaaSecurityConfig.class, IdentitySecurityConfig.class}) // don't activate mock users in case of productive config
@Conditional(MockUsersConfiguredCondition.class) // only load in case mock users are configured
@ConditionalOnWebApplication
@EnableWebSecurity
@Order(1000)
public class MockUsersSecurityConfig {

	private final static Logger logger = LoggerFactory.getLogger(MockUsersSecurityConfig.class);
	private final static ObjectMapper mapper = new ObjectMapper().setSerializationInclusion(Include.NON_NULL);

	@Autowired
	private HttpSecurityConfigurer httpSecurityConfigurer;

	@Autowired
	private CdsRuntime runtime;

	@Bean
	public SecurityFilterChain mockFilterChain(HttpSecurity http) throws Exception {
		http.csrf(c -> c.disable());
		httpSecurityConfigurer.configure(http);
		http.httpBasic(c -> {});

		Collection<User> users = runtime.getEnvironment().getCdsProperties().getSecurity().getMock().getUsers().values();
		if (users.isEmpty()) {
			logger.warn("No mock users found in configuration. Spring standard authentication manager with default user 'user' (password written to console) is available");
		} else {
			http.authenticationProvider(buildAuthenticationProvider(users));
		}

		logger.info("*************************************************************************");
		logger.info("*  Security configuration based on mock users found in active profile.  *");
		logger.info("*                 !!! NEVER USE IN PRODUCTIVE MODE !!!                  *");
		logger.info("*************************************************************************");

		return http.build();
	}

	private DaoAuthenticationProvider buildAuthenticationProvider(Collection<User> users) {
		List<UserDetails> userDetails = new ArrayList<>();
		for (User user : users) {
			if (!user.isValid()) {
				logger.warn("Skipping invalid mock user defintion {} ('name' and 'password' required)", stringifyUser(user));
			} else {
				userDetails.add(org.springframework.security.core.userdetails.User.builder()
					.username(user.getName())
					.password("{noop}" + user.getPassword())
					.authorities(user.getRoles().toArray(new String[0]))
					.build());
				logger.info("Added mock user {}", stringifyUser(user));
			}
		}

		UserDetailsService userDetailsService = new InMemoryUserDetailsManager(userDetails);
		DaoAuthenticationProvider authenticationProvider = new DaoAuthenticationProvider();
		authenticationProvider.setUserDetailsService(userDetailsService);
		return authenticationProvider;
	}

	private String stringifyUser(User user) {
		try {
			return mapper.writeValueAsString(user);
		} catch (Exception e) { // NOSONAR
			return user.getName();
		}
	}

}
