/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.framework.spring.config.auth.xsuaa;

import java.util.List;
import java.util.Properties;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.SpringApplication;
import org.springframework.boot.env.EnvironmentPostProcessor;
import org.springframework.core.env.ConfigurableEnvironment;
import org.springframework.core.env.PropertiesPropertySource;
import org.springframework.core.env.PropertySource;

import com.sap.cds.framework.spring.config.runtime.BootstrapCache;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.StringUtils;
import com.sap.cds.services.utils.XsuaaUtils;
import com.sap.cloud.environment.servicebinding.api.ServiceBinding;
import com.sap.cloud.security.xsuaa.XsuaaServiceConfiguration;
import com.sap.cloud.security.xsuaa.XsuaaServicePropertySourceFactory;

public class XsuaaEnvironmentPostProcessor implements EnvironmentPostProcessor {

	private static final Logger logger = LoggerFactory.getLogger(XsuaaEnvironmentPostProcessor.class);
	private static final String PROPERTYSOURCE_NAME = "cds-xsuaa";
	private static boolean xsuaaLibraryAvailable;

	static {
		try {
			xsuaaLibraryAvailable = XsuaaServiceConfiguration.class.getName() != null;
		} catch (NoClassDefFoundError e) { // NOSONAR
			xsuaaLibraryAvailable = false;
		}
	}


	@Override
	public void postProcessEnvironment(ConfigurableEnvironment environment, SpringApplication application) {
		if(!xsuaaLibraryAvailable) {
			return;
		}
		CdsRuntime runtime = BootstrapCache.get(environment).getCdsRuntime();
		XsuaaUtils xsuaaUtils = new XsuaaUtils(runtime);
		if (!xsuaaUtils.xsuaaAuthenticationConfigured()) {
			return;
		}
		environment.getPropertySources().addLast(createPropertySource(runtime));
	}

	private PropertySource<?> createPropertySource(CdsRuntime runtime) {
		XsuaaUtils xsuaaUtils = new XsuaaUtils(runtime);
		List<ServiceBinding> bindings = xsuaaUtils.getXsuaaServiceBindings();
		if(bindings.isEmpty()) {
			String xsuaaServiceBinding = runtime.getEnvironment().getCdsProperties().getSecurity().getXsuaa().getBinding();
			if (StringUtils.isEmpty(xsuaaServiceBinding)) {
				logger.info("No service binding with tag 'xsuaa' found");
			} else {
				logger.info("No service binding with tag 'xsuaa' and name '{}' found", xsuaaServiceBinding);
			}
			return new PropertiesPropertySource(PROPERTYSOURCE_NAME, new Properties());
		} else if (bindings.size() > 1) {
			logger.warn("Multiple service bindings with tag 'xsuaa' found");
		}

		ServiceBinding binding = bindings.get(0);
		logger.info("Using service binding '{}' to configure 'xsuaa' properties section", binding.getName().get());
		Properties props = new Properties();
		props.putAll(binding.getCredentials());

		return XsuaaServicePropertySourceFactory.create(PROPERTYSOURCE_NAME, props);
	}

}
