package com.sap.cds.framework.spring.utils;

import java.io.UncheckedIOException;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.WebApplicationType;
import org.springframework.boot.builder.SpringApplicationBuilder;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.sap.cds.services.mt.DeploymentService;
import com.sap.cds.services.mt.TenantProviderService;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.StringUtils;

public class MtCommand {

	private static final Logger logger = LoggerFactory.getLogger(MtCommand.class);
	private static final ObjectMapper mapper = new ObjectMapper();
	private static final String ENV_TENANTS = "MTCOMMAND_TENANTS"; // comma separated list
	private static final String ENV_OPTIONS = "MTCOMMAND_OPTIONS"; // JSON object

	static int execute(Class<?> clazz, Task task) {
		try {
			// start without web server
			CdsRuntime runtime = new SpringApplicationBuilder(clazz)
					.properties("cds.environment.command.enabled=true")
					.web(WebApplicationType.NONE).run(new String[0])
					.getBean(CdsRuntime.class);
			DeploymentService deploymentService = runtime.getServiceCatalog().getService(DeploymentService.class, DeploymentService.DEFAULT_NAME);
			assert runtime.getEnvironment().getCdsProperties().getEnvironment().getCommand().isEnabled();

			runtime.requestContext().privilegedUser().run(context -> {
				task.execute(deploymentService, runtime, System.currentTimeMillis());
			});

			logSuccessMessage();
			return 0;
		} catch (Throwable t) {
			logger.error("Failed to perform command '{}'", clazz.getSimpleName(), t);
			logErrorMessage();
			return 1;
		}
	}

	@FunctionalInterface
	static interface Task {

		void execute(DeploymentService deploymentService, CdsRuntime runtime, long startTime);

	}

	static String getTenant(String[] args) {
		String tenant;
		if (args.length > 0) {
			tenant = args[0];
		} else {
			tenant = System.getenv(ENV_TENANTS);
			if (tenant.contains(",")) {
				logger.error("Specify only one tenant");
				logErrorMessage();
				System.exit(2);
			}
		}

		if (StringUtils.isEmpty(tenant)) {
			logger.error("No tenant specified");
			logErrorMessage();
			System.exit(2);
		}

		return tenant;
	}

	static List<String> getTenants(String[] args, CdsRuntime runtime) {
		List<String> tenants;
		if (args.length > 0) {
			tenants = Arrays.asList(args);
		} else {
			String tenantsEnv = System.getenv(ENV_TENANTS);
			if (!StringUtils.isEmpty(tenantsEnv)) {
				tenants = Arrays.asList(tenantsEnv.split(","));
			} else {
				logger.info("Obtaining all subscribed tenants");
				TenantProviderService tenantService = runtime.getServiceCatalog().getService(TenantProviderService.class, TenantProviderService.DEFAULT_NAME);
				tenants = tenantService.readTenants();
			}
		}

		return tenants;
	}

	static Map<String, Object> getOptions(String[] args, boolean optionsOnCommandline) {
		try {
			Map<String, Object> options;
			if (optionsOnCommandline && args.length > 1) {
				options = mapper.readValue(args[1], new TypeReference<Map<String,Object>>(){});
			} else {
				String optionsEnvVar = System.getenv(ENV_OPTIONS);
				if (optionsEnvVar != null) {
					options = mapper.readValue(optionsEnvVar, new TypeReference<Map<String,Object>>(){});
				} else {
					options = new HashMap<>();
				}
			}
			return options;
		} catch (JsonProcessingException e) {
			throw new UncheckedIOException(e);
		}
	}

	private static void logSuccessMessage() {
		logger.info("*************");
		logger.info("*  SUCCESS  *");
		logger.info("*************");
	}

	private static void logErrorMessage() {
		logger.error("***********");
		logger.error("*  ERROR  *");
		logger.error("***********");
	}
}
