package com.sap.cds.framework.spring.config.auth;

import static org.springframework.security.web.util.matcher.AntPathRequestMatcher.antMatcher;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Stream;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.autoconfigure.AutoConfiguration;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.context.annotation.Bean;
import org.springframework.http.HttpMethod;
import org.springframework.security.config.annotation.web.WebSecurityConfigurer;
import org.springframework.security.config.annotation.web.builders.HttpSecurity;
import org.springframework.security.web.util.matcher.AntPathRequestMatcher;
import org.springframework.security.web.util.matcher.RequestMatcher;

import com.sap.cds.adapter.ServletUrlResourcePaths;
import com.sap.cds.adapter.ServletUrlResourcePaths.UrlResourcePathVisitor;
import com.sap.cds.adapter.UrlResourcePath;
import com.sap.cds.services.runtime.CdsRuntime;

@AutoConfiguration
@ConditionalOnClass(WebSecurityConfigurer.class) // only if spring-security library has been loaded
public class CdsHttpSecurityConfigurerConfig {

	@Bean
	@ConditionalOnMissingBean  // Auto configuration classes are guaranteed to be loaded after any user-defined bean definitions, hence no ordering is required
	public HttpSecurityConfigurer httpSecurityConfigurer(CdsRuntime runtime) {
		return new CdsModelBasedSecurityConfigurer(runtime);
	}

	/**
	 * An HttpSecurityConfigurer which is based on the service model of the underlying CDS model.
	 * Endpoints that are not authorized with @requires or @restrict are not authenticated by default.
	 */
	private static class CdsModelBasedSecurityConfigurer implements HttpSecurityConfigurer {

		private static final Logger logger = LoggerFactory.getLogger(CdsModelBasedSecurityConfigurer.class);

		private final CdsRuntime runtime;

		public CdsModelBasedSecurityConfigurer(CdsRuntime runtime) {
			this.runtime = runtime;
		}

		@Override
		public void configure(HttpSecurity http) throws Exception { // NOSONAR

			ServletUrlResourcePaths servletPaths = new ServletUrlResourcePaths(runtime);

			if( !runtime.getEnvironment().getCdsProperties().getSecurity().getAuthentication().isAuthenticateUnknownEndpoints() ) {
				// narrow the base urls to the urls of the servlets
				http.securityMatchers(matcher -> matcher.requestMatchers(servletPaths.getBasePaths().map(UrlResourcePath::getPath).map(AntPathRequestMatcher::antMatcher).toArray(RequestMatcher[]::new)));
				logger.info("Configuring authentication of CDS adapter endpoints. Other endpoints are not configured.");
			}

			// collect all open open endpoints which are declared as public
			List<AntPathRequestMatcher> publicPathMatchers = new ArrayList<>();

			http.authorizeHttpRequests(urlRegistry -> {
				servletPaths.visit(new UrlResourcePathVisitor() {
					@Override
					public void foundPublicPath(UrlResourcePath publicPath) {
						publicPathMatchers.add(antMatcher(publicPath.getPath()));
						urlRegistry.requestMatchers(antMatcher(publicPath.getPath())).permitAll();
						logger.debug("Public CDS endpoint {}", publicPath.getPath());
					}

					@Override
					public void foundPublicEvents(UrlResourcePath path, Stream<String> publicEvents) {
						publicEvents.forEach(publicEvent -> {
							publicPathMatchers.add(antMatcher(HttpMethod.valueOf(publicEvent),
									path.getPath()));
							urlRegistry.requestMatchers(antMatcher(HttpMethod.valueOf(publicEvent),
									path.getPath())).permitAll();
							logger.debug("Public CDS endpoint for method {} {}", publicEvent, path.getPath());
						});

						urlRegistry.requestMatchers(antMatcher(path.getPath())).authenticated(); // Required as parent path may be open with recursive pattern
						logger.debug("Authenticate CDS endpoint {}", path.getPath());
					}
				});
				// all other endpoints of the servlets are closed by default
				urlRegistry.anyRequest().authenticated();
			});

			http.csrf(c -> c.ignoringRequestMatchers(publicPathMatchers.toArray(new AntPathRequestMatcher[0]))); // no csrf for public endpoints
		}
	}

}