package com.sap.cds.framework.spring.config.runtime;

import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.sql.DataSource;

import org.springframework.core.env.Environment;

import com.sap.cds.adapter.AdapterFactory;
import com.sap.cds.services.datasource.DataSourceDescriptor;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.runtime.CdsRuntimeConfigurer;
import com.sap.cds.services.runtime.ExtendedServiceLoader;
import com.sap.cds.services.utils.datasource.DataSourceUtils;

/**
 * Caches the {@link CdsRuntimeConfigurer} during initialization of the Spring Boot Application Context.
 */
public class BootstrapCache {

	private static final Map<Environment, BootstrapCache> cache = Collections.synchronizedMap(new HashMap<>());

	private final CdsRuntimeConfigurer configurer;

	private boolean servicesInitialized;
	private Map<String, AdapterFactory> adapterFactories;
	private List<DataSourceDescriptor> dataSourceDescriptors;
	private Map<String, DataSource> dataSources;

	public static BootstrapCache get(Environment environment) {
		return cache.computeIfAbsent(environment, BootstrapCache::new);
	}

	public static void clear(Environment environment) {
		cache.remove(environment);
	}

	private BootstrapCache(Environment environment) {
		this.configurer = CdsRuntimeConfigurer.create(new SpringPropertiesProvider(environment)).environmentConfigurations();
	}

	public BootstrapCache ensureServices() {
		if (!servicesInitialized) {
			configurer.cdsModel().serviceConfigurations();
			servicesInitialized = true;
		}
		return this;
	}

	public CdsRuntimeConfigurer getConfigurer() {
		return configurer;
	}

	public CdsRuntime getCdsRuntime() {
		return configurer.getCdsRuntime();
	}

	public Map<String, AdapterFactory> getAdapterFactories() {
		if (adapterFactories == null) {
			ensureServices();
			Map<String, AdapterFactory> factories = new HashMap<>();
			ExtendedServiceLoader.loadAll(AdapterFactory.class, configurer.getCdsRuntime()).forEachRemaining(f -> factories.put(f.getClass().getSimpleName(), f));
			adapterFactories = Collections.unmodifiableMap(factories);
		}
		return adapterFactories;
	}

	public List<DataSourceDescriptor> getDataSourceDescriptors() {
		if (dataSourceDescriptors == null) {
			dataSourceDescriptors = Collections.unmodifiableList(DataSourceUtils.getDataSourceDescriptors(configurer.getCdsRuntime()));
		}
		return dataSourceDescriptors;
	}

	public Map<String, DataSource> getDataSources() {
		if (dataSources == null) {
			dataSources = Collections.unmodifiableMap(DataSourceUtils.getDataSources(configurer.getCdsRuntime()));
		}
		return dataSources;
	}

}
