package com.sap.cds.framework.spring.config.aot;

import java.util.Arrays;

import org.springframework.aot.hint.ExecutableMode;
import org.springframework.aot.hint.RuntimeHints;
import org.springframework.beans.factory.aot.BeanFactoryInitializationAotContribution;
import org.springframework.beans.factory.aot.BeanFactoryInitializationAotProcessor;
import org.springframework.beans.factory.config.ConfigurableListableBeanFactory;
import org.springframework.transaction.PlatformTransactionManager;
import org.springframework.util.ReflectionUtils;

import com.sap.cds.framework.spring.config.datasource.PlatformTransactionManagerPostProcessor;
import com.sap.cds.services.environment.CdsProperties;
import com.sap.cds.services.runtime.CdsRuntimeConfigurer;

public class CdsBeanFactoryAwareRuntimeHints implements BeanFactoryInitializationAotProcessor {

	private static boolean springTxAvailable;

	static {
		try {
			springTxAvailable = PlatformTransactionManager.class.getName() != null;
		} catch (NoClassDefFoundError e) { // NOSONAR
			springTxAvailable = false;
		}
	}

	@Override
	@SuppressWarnings("unchecked")
	public BeanFactoryInitializationAotContribution processAheadOfTime(ConfigurableListableBeanFactory beanFactory) {
		return (generationContext, beanFactoryInitializationCode) -> {
			// Known Issues
			// https://github.com/spring-projects/spring-boot/issues/36030
			// https://github.com/oracle/graal/issues/6457

			RuntimeHints hints = generationContext.getRuntimeHints();
			CdsRuntimeConfigurer configurer = beanFactory.getBean(CdsRuntimeConfigurer.class);

			// resources
			CdsProperties properties = configurer.getCdsRuntime().getEnvironment().getCdsProperties();
			hints.resources()
				.registerPattern(properties.getModel().getCsnPath())
				.registerPattern(properties.getOdataV4().getEdmxPath() + "/.*")
				.registerPattern(properties.getOdataV2().getEdmxPath() + "/.*");

			// PlatformTransactionManager proxies
			if (springTxAvailable) {
				Arrays.stream(beanFactory.getBeanNamesForType(PlatformTransactionManager.class))
					.map(name -> (Class<Object>) beanFactory.getType(name))
					.forEach(clazz -> {
						hints.proxies().registerJdkProxy(PlatformTransactionManagerPostProcessor.determineProxyInterfaces(clazz));
					});
			}

			hints.reflection().registerMethod(ReflectionUtils.findMethod(String.class, "toUpperCase"), ExecutableMode.INVOKE);
		};
	}

}
