/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.framework.spring.config.runtime;

import java.util.ArrayDeque;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collector;
import java.util.stream.Stream;

import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.AutoConfiguration;
import org.springframework.context.ApplicationContext;
import org.springframework.core.annotation.AnnotationAwareOrderComparator;
import org.springframework.core.env.Environment;

import com.sap.cds.services.Service;
import com.sap.cds.services.ServiceCatalog;
import com.sap.cds.services.application.ApplicationLifecycleService;
import com.sap.cds.services.environment.ApplicationInfoProvider;
import com.sap.cds.services.environment.ServiceBindingProvider;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.runtime.CdsProvider;
import com.sap.cds.services.runtime.CdsRuntimeConfigurer;

/**
 * Bean performs the initialization and stopping of the CdsRuntime.
 */
@AutoConfiguration(after = CdsRuntimeBeanDefinitionRegistrar.class)
public class CdsRuntimeInitializer implements InitializingBean, DisposableBean {

	@Autowired
	private CdsRuntimeConfigurer configurer;

	@Autowired
	private List<Service> services;

	@Autowired(required = false)
	private List<CdsProvider<?>> providers;

	@Autowired
	private ApplicationContext context;

	@Autowired
	private Environment environment;

	@Override
	@SuppressWarnings({"unchecked", "rawtypes"})
	public void afterPropertiesSet() throws Exception {
		// ensure that all services are registered
		ServiceCatalog catalog = configurer.getCdsRuntime().getServiceCatalog();
		services.stream().filter(s -> catalog.getService(s.getName()) == null).forEach(configurer::service);

		// register event handlers
		configurer.eventHandlerConfigurations();

		// register custom event handlers
		Arrays.stream(context.getBeanNamesForType(EventHandler.class))
		// filter all beans which types have been duplicated through scoping by CGLIB
		.filter(name -> !name.startsWith("scopedTarget."))
		.map(context::getType).distinct()
		.sorted(AnnotationAwareOrderComparator.INSTANCE)
		.forEach((handlerClass) -> {
			configurer.eventHandler((Class) handlerClass, () -> context.getBean(handlerClass));
		});

		// register providers
		configurer.providerConfigurations();

		// register custom providers. The lists are in sorted order induced by @Order annotation.
		if (providers != null) {
			reverse(providers.stream())
				.filter(p -> !ServiceBindingProvider.class.isInstance(p) && !ApplicationInfoProvider.class.isInstance(p))
				.forEach(p -> configurer.provider((CdsProvider<?>) p));
		}

		// lock cds runtime configurer
		configurer.complete();
		BootstrapCache.clear(environment);

		/*
		 * fire runtime started event
		 *
		 * - after all beans have been initialized and before web server is up and running
		 * - in the initialization phase we want to have the base model plus all toggles from the model provide. For
		 *   that reason we need to create a request context with a wild card toggle.
		 */
		configurer.getCdsRuntime().requestContext().systemUserProvider().run(req -> {
			catalog.getServices(ApplicationLifecycleService.class).forEach(as -> as.applicationPrepared());
		});
	}

	@Override
	public void destroy() throws Exception {
		configurer.getCdsRuntime().requestContext().systemUserProvider().run(req -> {
			configurer.getCdsRuntime().getServiceCatalog()
					.getServices(ApplicationLifecycleService.class)
					.forEach(as -> as.applicationStopped());
		});
	}

	/**
	 * Creates a reverse stream.
	 *
	 * @param stream The stream to be put in reverse order
	 * @return The stream in reverse order
	 */
	private static <T> Stream<T> reverse(Stream<T> stream) {
		return stream.collect(Collector.of(() -> new ArrayDeque<T>(), ArrayDeque::addFirst, (q1, q2) -> {
			q2.addAll(q1);
			return q2;
		})).stream();
	}

}
