/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.framework.spring.config.auth.xsuaa;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.AutoConfiguration;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnWebApplication;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Conditional;
import org.springframework.core.annotation.Order;
import org.springframework.core.convert.converter.Converter;
import org.springframework.security.authentication.AbstractAuthenticationToken;
import org.springframework.security.config.annotation.web.builders.HttpSecurity;
import org.springframework.security.config.annotation.web.configuration.EnableWebSecurity;
import org.springframework.security.config.http.SessionCreationPolicy;
import org.springframework.security.oauth2.jwt.Jwt;
import org.springframework.security.web.SecurityFilterChain;

import com.sap.cds.framework.spring.config.auth.HttpSecurityConfigurer;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.XsuaaUtils;
import com.sap.cds.services.utils.XsuaaUtils.AuthenticationType;
import com.sap.cloud.security.xsuaa.XsuaaServiceConfiguration;
import com.sap.cloud.security.xsuaa.token.TokenAuthenticationConverter;

/**
 * The default security configuration in case of active XSUAA configuration - which should be the case in productive scenarios.
 * By default, all requests require an oauth2 authentication - including public endpoints.
 * In order to open public endpoints, the application needs to create a security configuration with higher priority
 * and override these endpoints.
 */
@AutoConfiguration
@ConditionalOnClass({ XsuaaServiceConfiguration.class, HttpSecurity.class, Jwt.class }) // only if spring-xsuaa and spring-security web & JWT library has been loaded
@Conditional(XsuaaConfiguredCondition.class)
@ConditionalOnWebApplication
@EnableWebSecurity
@Order(1000)
public class XsuaaSecurityConfig {

	private final static Logger logger = LoggerFactory.getLogger(XsuaaSecurityConfig.class);

	@Autowired
	private CdsRuntime runtime;

	@Autowired
	private XsuaaServiceConfiguration xsuaaServiceConfiguration;

	@Autowired
	private HttpSecurityConfigurer httpSecurityConfigurer;

	@Autowired
	private XsuaaFeature xsuaaFeatue;

	@Bean
	public SecurityFilterChain xsuaaFilterChain(HttpSecurity http) throws Exception {

		if ( !xsuaaFeatue.isActive() ) {
			throw new IllegalStateException("XSUAA configuration requires dependency to cds-feature-xsuaa. "
					+ "Alternatively, deactivate this security configuration with property 'cds.security.xsuaa.enabled: false' AND provide your own configuration.");
		}

		AuthenticationType authType = new XsuaaUtils(runtime).configuredAuthenticationType();

		// @formatter:off
		http.sessionManagement(session -> session.sessionCreationPolicy(SessionCreationPolicy.STATELESS)); // state is kept in AppRouter

		httpSecurityConfigurer.configure(http);

		http.oauth2ResourceServer(resourceServer -> resourceServer.jwt(customizer -> customizer.jwtAuthenticationConverter(getJwtAuthoritiesConverter())));
		// @formatter:on

		switch(authType) {
		case XSUAA:
			logger.info("Activated XSUAA security configuration");
			break;
		default:
			throw new IllegalStateException("Configuration error: Can't configure authentication without a binding to XSUAA service"); // this should not happen
		}

		return http.build();
	}

	private Converter<Jwt, AbstractAuthenticationToken> getJwtAuthoritiesConverter() {
		TokenAuthenticationConverter converter = new TokenAuthenticationConverter(xsuaaServiceConfiguration);
		converter.setLocalScopeAsAuthorities(true);
		return converter;
	}

}
