/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.framework.spring.config.runtime;

import java.sql.Connection;
import java.sql.SQLException;
import java.util.function.Supplier;

import javax.sql.DataSource;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.aot.hint.RuntimeHints;
import org.springframework.aot.hint.RuntimeHintsRegistrar;
import org.springframework.boot.autoconfigure.AutoConfiguration;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.jdbc.DataSourceAutoConfiguration;
import org.springframework.boot.autoconfigure.jdbc.DataSourceTransactionManagerAutoConfiguration;
import org.springframework.boot.jdbc.EmbeddedDatabaseConnection;
import org.springframework.boot.sql.init.dependency.DependsOnDatabaseInitialization;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.ImportRuntimeHints;
import org.springframework.context.annotation.Primary;
import org.springframework.jdbc.datasource.DataSourceTransactionManager;
import org.springframework.jdbc.datasource.TransactionAwareDataSourceProxy;
import org.springframework.transaction.PlatformTransactionManager;

import com.sap.cds.CdsDataStore;
import com.sap.cds.framework.spring.config.datasource.PlatformTransactionManagerPostProcessor.SpringTransactionManagerGetter;
import com.sap.cds.framework.spring.config.runtime.SpringJdbcPersistenceServiceConfiguration.SpringJdbcPersistenceServiceRuntimeHints;
import com.sap.cds.framework.spring.utils.ProxyUtils;
import com.sap.cds.services.impl.persistence.JdbcPersistenceService;
import com.sap.cds.services.impl.persistence.JdbcPersistenceServiceConfiguration;
import com.sap.cds.services.persistence.PersistenceService;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.runtime.CdsRuntimeConfigurer;
import com.sap.cds.services.transaction.TransactionManager;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;

@AutoConfiguration(after = { DataSourceAutoConfiguration.class, DataSourceTransactionManagerAutoConfiguration.class})
// only if cds-feature-jdbc, spring-jdbc are available
@ConditionalOnClass({ JdbcPersistenceServiceConfiguration.class, TransactionAwareDataSourceProxy.class })
@ConditionalOnBean({ DataSource.class, PlatformTransactionManager.class }) // datasource and tx manager must exist
@DependsOnDatabaseInitialization
@ImportRuntimeHints(SpringJdbcPersistenceServiceRuntimeHints.class)
public class SpringJdbcPersistenceServiceConfiguration {

	static class SpringJdbcPersistenceServiceRuntimeHints implements RuntimeHintsRegistrar {

		@Override
		public void registerHints(RuntimeHints hints, ClassLoader classLoader) {
			hints.proxies().registerJdkProxy(CdsDataStore.class);
		}

	}

	private static final Logger logger = LoggerFactory.getLogger(SpringJdbcPersistenceServiceConfiguration.class);
	private final JdbcPersistenceServiceConfiguration configuration;

	public static JdbcPersistenceServiceConfiguration createJdbcPersistenceServiceConfiguration(String serviceName, DataSource dataSource, PlatformTransactionManager platformTransactionManager) {
		TransactionManager txMgr = ((SpringTransactionManagerGetter) platformTransactionManager).getSpringTransactionManager(serviceName);
		DataSource txAwareDataSource = new TransactionAwareDataSourceProxy(dataSource);
		Supplier<Connection> connectionSupplier = () -> {
			try {
				return txAwareDataSource.getConnection();
			} catch (SQLException e) {
				throw new ErrorStatusException(CdsErrorStatuses.JDBC_CONNECTION_FAILED, e);
			}
		};
		return JdbcPersistenceServiceConfiguration.create(serviceName, connectionSupplier, txMgr);
	}

	public SpringJdbcPersistenceServiceConfiguration(DataSource dataSource, PlatformTransactionManager platformTransactionManager, CdsRuntime runtime) {
		// determine if datasource is considered embedded by Spring, if CSV initialization mode is set to embedded
		// the check attempts to create a datasource connection
		// embedded databases can not be auto-configured based on Service Bindings as done in CdsRuntimeConfig
		String csvInitMode = runtime.getEnvironment().getCdsProperties().getDataSource().getCsv().getInitializationMode();
		if("embedded".equals(csvInitMode) && isEmbedded(dataSource)) {
			logger.debug("Determined DataSource as embedded based on connection check.");
			runtime.getEnvironment().getCdsProperties().getDataSource().setEmbedded(true);
		}

		// checks that primary txMgr and DataSource work together
		PlatformTransactionManager originalPlatformTransactionManager = ((SpringTransactionManagerGetter) platformTransactionManager).getPlatformTransactionManager();
		if (originalPlatformTransactionManager instanceof DataSourceTransactionManager) {
			DataSourceTransactionManager dataSourceTransactionManager = (DataSourceTransactionManager) originalPlatformTransactionManager;
			if (dataSourceTransactionManager.getDataSource() != dataSource) {
				logger.warn("DataSource managed by primary DataSourceTransactionManager does not match primary DataSource");
			}
		} else {
			logger.warn("Primary PlatformTransactionManager is not a DataSourceTransactionManager but has type '{}'", originalPlatformTransactionManager.getClass());
		}

		this.configuration = createJdbcPersistenceServiceConfiguration(PersistenceService.DEFAULT_NAME, dataSource, platformTransactionManager);
	}

	@Primary
	@Bean(PersistenceService.DEFAULT_NAME)
	public JdbcPersistenceService persistenceService(CdsRuntimeConfigurer configurer) {
		return configuration.createOrGetService(configurer);
	}

	@Bean
	public CdsDataStore cdsDataStore(CdsRuntimeConfigurer configurer) {
		JdbcPersistenceService persistenceService = configuration.createOrGetService(configurer);
		return ProxyUtils.createProxy(CdsDataStore.class, () -> persistenceService.getCdsDataStore());
	}

	private boolean isEmbedded(DataSource dataSource) {
		try {
			return EmbeddedDatabaseConnection.isEmbedded(dataSource);
		} catch (Exception e) {
			return false;
		}
	}

}
