/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.framework.spring.config.runtime;

import java.util.Map;

import org.springframework.boot.context.properties.bind.BindResult;
import org.springframework.boot.context.properties.bind.Bindable;
import org.springframework.boot.context.properties.bind.Binder;
import org.springframework.core.env.Environment;

import com.sap.cds.services.environment.PropertiesProvider;

/**
 * A {@code PropertiesProvider} exposing access to all Spring properties.
 */
public class SpringPropertiesProvider implements PropertiesProvider {

	private final Environment environment;

	public SpringPropertiesProvider(Environment environment) {
		this.environment = environment;
	}

	@Override
	public <T> T bindPropertyClass(String section, Class<T> clazz) {
		BindResult<T> bindResult = Binder.get(environment).bind(section, Bindable.of(clazz));
		if (bindResult.isBound()) {
			return bindResult.get();
		} else {
			try {
				return clazz.getDeclaredConstructor().newInstance();
			} catch (ReflectiveOperationException e) { // NOSONAR
				return null;
			}
		}
	}

	@Override
	public String getProperty(String key, String defaultValue) {
		return environment.getProperty(key, defaultValue);
	}

	@Override
	public <T> T getProperty(String key, Class<T> clazz, T defaultValue) {
		if(Map.class.isAssignableFrom(clazz)) {
			return bindPropertyClass(key, clazz);
		}
		return environment.getProperty(key, clazz, defaultValue);
	}

}


