/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.framework.spring.config.auth.identity;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.SpringApplication;
import org.springframework.boot.env.EnvironmentPostProcessor;
import org.springframework.core.env.ConfigurableEnvironment;
import org.springframework.core.env.PropertiesPropertySource;
import org.springframework.core.env.PropertySource;

import com.sap.cds.framework.spring.config.runtime.BootstrapCache;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.IdentityUtils;
import com.sap.cloud.environment.servicebinding.api.ServiceBinding;
import com.sap.cloud.security.config.Environments;
import com.sap.cloud.security.config.ServiceBindingEnvironment;
import com.sap.cloud.security.spring.config.IdentityServiceConfiguration;

public class IdentityEnvironmentPostProcessor implements EnvironmentPostProcessor {

	private final static Logger logger = LoggerFactory.getLogger(IdentityEnvironmentPostProcessor.class);
	private static final String PROPERTYSOURCE_NAME = "cds-identity";
	private static final String PARAM_PROOF_TOKEN = "sap.spring.security.identity.prooftoken";

	private static boolean identityLibraryAvailable;

	static {
		try {
			identityLibraryAvailable = IdentityServiceConfiguration.class.getName() != null;
		} catch (NoClassDefFoundError e) { // NOSONAR
			identityLibraryAvailable = false;
		}
	}

	@Override
	public void postProcessEnvironment(ConfigurableEnvironment environment, SpringApplication application) {
		if(!identityLibraryAvailable) {
			return;
		}

		CdsRuntime runtime = BootstrapCache.get(environment).getCdsRuntime();

		IdentityUtils identityUtils = new IdentityUtils(runtime);

		List<ServiceBinding> identityBindings = identityUtils.getIasServiceBindings(); // IAS takes precedence
		if (identityBindings.size() > 1) {
			logger.warn("Multiple service bindings with tag 'identity' found");
		}

		List<ServiceBinding> xsuaaBindings = identityUtils.getXsuaaServiceBindings();
		if (xsuaaBindings.size() > 1) {
			logger.warn("Multiple service bindings with tag 'xsuaa' found");
		}

		List<ServiceBinding> bindings = new ArrayList<>();
		if(!identityBindings.isEmpty()) {
			bindings.add(identityBindings.get(0));

			if (!environment.containsProperty(PARAM_PROOF_TOKEN)) {
				environment.getPropertySources().addLast(createPropertySource(runtime));
			}
		}

		if(!xsuaaBindings.isEmpty()) {
			bindings.add(xsuaaBindings.get(0));
		}

		bindings.forEach(b -> logger.info("Using service binding '{}' to configure identity authentication", b.getName().get()));


		// TODO: Using reflection as dirty workaround here until java-security provides an API to control the Environment
		ServiceBindingEnvironment env = new ServiceBindingEnvironment(() -> bindings);
		try {
			Field field = Environments.class.getDeclaredField("currentEnvironment");
			field.setAccessible(true);
			field.set(null, env);
		} catch (Exception e) {
			throw new InternalError("Missing field com.sap.cloud.security.config.Environments.currentEnvironment", e);
		}
	}

	private PropertySource<?> createPropertySource(CdsRuntime runtime) {
		Properties props = new Properties();
		props.put(PARAM_PROOF_TOKEN, true);

		return new PropertiesPropertySource(PROPERTYSOURCE_NAME, props);
	}

}
