/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.framework.spring.config.auth.identity;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.AutoConfiguration;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnWebApplication;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Conditional;
import org.springframework.context.annotation.PropertySource;
import org.springframework.core.annotation.Order;
import org.springframework.core.convert.converter.Converter;
import org.springframework.security.authentication.AbstractAuthenticationToken;
import org.springframework.security.config.annotation.web.builders.HttpSecurity;
import org.springframework.security.config.annotation.web.configuration.EnableWebSecurity;
import org.springframework.security.config.http.SessionCreationPolicy;
import org.springframework.security.oauth2.jwt.Jwt;
import org.springframework.security.web.SecurityFilterChain;

import com.sap.cds.framework.spring.config.auth.HttpSecurityConfigurer;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.IdentityUtils;
import com.sap.cds.services.utils.IdentityUtils.AuthenticationType;
import com.sap.cloud.security.spring.config.IdentityServiceConfiguration;
import com.sap.cloud.security.spring.config.IdentityServicesPropertySourceFactory;



/**
 * The default security configuration in case of active IAS or XSUAA configuration - which should be the case in productive scenarios.
 * By default, all requests require an oauth2 authentication - including public endpoints.
 * In order to open public endpoints, the application needs to create a security configuration with higher priority
 * and override these endpoints.
 */
@AutoConfiguration
@ConditionalOnClass({ IdentityServiceConfiguration.class, HttpSecurity.class, Jwt.class }) // only if spring-security and spring-security-web & JWT library has been loaded
@Conditional(IdentityConfiguredCondition.class)
@PropertySource(factory = IdentityServicesPropertySourceFactory.class, value = {""})
@ConditionalOnWebApplication
@EnableWebSecurity
@Order(1000)
public class IdentitySecurityConfig {

	private final static Logger logger = LoggerFactory.getLogger(IdentitySecurityConfig.class);

	@Autowired
	private CdsRuntime runtime;

	@Autowired
	private HttpSecurityConfigurer httpSecurityConfigurer;

	@Autowired
	private IdentityFeature identityFeatue;

	@Autowired(required = false)
	private Converter<Jwt, AbstractAuthenticationToken> authConverter; // required in case of xsuaa only

	@Bean
	public SecurityFilterChain identityFilterChain(HttpSecurity http) throws Exception {
		if (!identityFeatue.isActive()) {
			throw new IllegalStateException(
					"""
					Identity configuration requires dependency to cds-feature-identity. \
					Alternatively, deactivate this security configuration with property 'cds.security.identity.enabled: false' AND provide your own configuration.\
					""");
		}

		AuthenticationType authType = new IdentityUtils(runtime).configuredAuthenticationType();

		// @formatter:off
		http.sessionManagement(s -> s.sessionCreationPolicy(SessionCreationPolicy.STATELESS)); // state is kept in AppRouter

		httpSecurityConfigurer.configure(http);

		http.oauth2ResourceServer(resourceServer -> resourceServer.jwt(jwtConfigurer -> {
			if(authConverter != null) {
				jwtConfigurer.jwtAuthenticationConverter(authConverter);
			}
		}));

		// @formatter:on

		switch(authType) {
		case XSUAA:
			if (authConverter == null) {
				throw new IllegalStateException("Identity configuration for XSUAA requires Converter<Jwt, AbstractAuthenticationToken>");
			}
			logger.info("Activated identity security configuration based on XSUAA");
			break;
		case IAS:
			logger.info("Activated identity security configuration based on IAS");
			break;
		case IAS_XSUAA:
			if (authConverter == null) {
				throw new IllegalStateException("Identity configuration for XSUAA requires Converter<Jwt, AbstractAuthenticationToken>");
			}
			logger.info("Activated identity security configuration based on IAS and XSUAA");
			break;
		default:
			throw new IllegalStateException("Configuration error: Can't configure authentication without a unique binding to either XSUAA or IAS service");
		}
		return http.build();
	}

}
