/*********************************************************************
 * (C) 2024 SAP SE or an SAP affiliate company. All rights reserved. *
 *********************************************************************/
package com.sap.cds.framework.spring.actuator;

import java.time.Duration;
import java.time.Instant;

import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.util.EntityUtils;
import org.springframework.boot.actuate.health.Health;
import org.springframework.boot.actuate.health.HealthIndicator;

import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cloud.sdk.cloudplatform.connectivity.DestinationAccessor;
import com.sap.cloud.sdk.cloudplatform.connectivity.HttpClientAccessor;
import com.sap.cloud.sdk.cloudplatform.resilience.ResilienceConfiguration;
import com.sap.cloud.sdk.cloudplatform.resilience.ResilienceConfiguration.TimeLimiterConfiguration;
import com.sap.cloud.sdk.cloudplatform.resilience.ResilienceDecorator;

class DestinationBasedHealthIndicator implements HealthIndicator {

	private static final Duration TIMEOUT_DURATION = Duration.ofSeconds(1);
	private static final Duration STATUS_TIME_TO_LIVE = Duration.ofSeconds(10);

	private final CdsRuntime runtime;
	private final String destinationName;
	private HttpClient httpClient;
	private Health lastHealth;
	private Instant lastCheckedAt;
	private final ResilienceConfiguration resilienceConfiguration;

	DestinationBasedHealthIndicator(CdsRuntime runtime, String destinationName) {
		this.runtime = runtime;
		this.destinationName = destinationName;
		this.resilienceConfiguration = ResilienceConfiguration.of(destinationName + ".health")
				.timeLimiterConfiguration(TimeLimiterConfiguration.of(TIMEOUT_DURATION));
	}

	@Override
	public Health health() {
		// NB: Destination created outside of health indicator, but might not be available in the constructor
		if (httpClient == null) {
			httpClient = HttpClientAccessor.getHttpClient(DestinationAccessor.getDestination(destinationName).asHttp());
		}

		Instant now = Instant.now();
		if (lastCheckedAt == null || lastCheckedAt.plus(STATUS_TIME_TO_LIVE).isBefore(now)) {
			lastCheckedAt = now;

			lastHealth = runtime.requestContext().systemUserProvider().run(r -> {
				try {
					return ResilienceDecorator.executeCallable(() -> {
						HttpResponse response = httpClient.execute(new HttpGet("/health"));
						// No response expected from the sidecar
						EntityUtils.consumeQuietly(response.getEntity());
						if (HttpStatus.SC_OK == response.getStatusLine().getStatusCode()) {
							return Health.up().build();
						}
						return Health.down().build();
					}, resilienceConfiguration);
				} catch (Exception e) {
					return Health.down().build();
				}
			});
		}
		return lastHealth;
	}
}
