/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.framework.spring.actuator;

import org.springframework.boot.actuate.autoconfigure.health.ConditionalOnEnabledHealthIndicator;
import org.springframework.boot.actuate.autoconfigure.jdbc.DataSourceHealthContributorAutoConfiguration;
import org.springframework.boot.actuate.health.HealthContributor;
import org.springframework.boot.actuate.health.HealthIndicator;
import org.springframework.boot.autoconfigure.AutoConfiguration;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnSingleCandidate;
import org.springframework.boot.autoconfigure.jdbc.DataSourceAutoConfiguration;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Conditional;

import com.sap.cds.feature.mt.RoutingDataSourceFactory;
import com.sap.cds.framework.spring.config.runtime.CdsRuntimeBeanDefinitionRegistrar;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.feature.mt.lib.runtime.TenantAwareDataSource;


/**
 * Configuration which overrides the standard Spring DB health check, if a {@link TenantAwareDataSource} is present.
 */
@AutoConfiguration(before = DataSourceHealthContributorAutoConfiguration.class, after = { DataSourceAutoConfiguration.class, CdsRuntimeBeanDefinitionRegistrar.class })
@ConditionalOnClass({HealthIndicator.class, RoutingDataSourceFactory.class}) // spring-actuator + cds-features-mt must be in classpath
@ConditionalOnSingleCandidate(TenantAwareDataSource.class) // a (primary) TenantAwareDataSource bean must exist
@Conditional(MtHealthCheckCondition.class)
@ConditionalOnEnabledHealthIndicator("db")
public class TenantAwareDataSourceHealthIndicatorConfig {

	@Bean("dbHealthIndicator") // this will ensure to override the bean from DataSourceHealthContributorAutoConfiguration
	public HealthContributor dbHealthIndicatorTenantAware(TenantAwareDataSource dataSource, CdsRuntime runtime) {
		return new TenantAwareDataSourceHealthIndicator(dataSource, runtime);
	}

}
