/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.framework.spring.config.runtime;

import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Set;

import javax.sql.DataSource;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.FactoryBean;
import org.springframework.beans.factory.support.AbstractBeanDefinition;
import org.springframework.beans.factory.support.BeanDefinitionBuilder;
import org.springframework.beans.factory.support.BeanDefinitionRegistry;
import org.springframework.context.EnvironmentAware;
import org.springframework.context.annotation.ImportBeanDefinitionRegistrar;
import org.springframework.core.env.Environment;
import org.springframework.core.type.AnnotationMetadata;
import org.springframework.jdbc.datasource.DataSourceTransactionManager;
import org.springframework.web.servlet.handler.SimpleUrlHandlerMapping;

import com.sap.cds.adapter.AdapterFactory;
import com.sap.cds.adapter.ServletAdapterFactory;
import com.sap.cds.framework.spring.config.adapter.AdapterBeanFactory;
import com.sap.cds.framework.spring.config.datasource.DataSourceBeanFactory;
import com.sap.cds.framework.spring.config.datasource.DataSourceDescriptorBeanFactory;
import com.sap.cds.framework.spring.config.datasource.DataSourceTransactionManagerBeanFactory;
import com.sap.cds.framework.spring.config.datasource.JdbcPersistenceServiceBeanFactory;
import com.sap.cds.services.cds.ApplicationService;
import com.sap.cds.services.cds.RemoteService;
import com.sap.cds.services.datasource.DataSourceDescriptor;
import com.sap.cds.services.environment.CdsProperties;
import com.sap.cds.services.environment.CdsProperties.Persistence.PersistenceServiceConfig;
import com.sap.cds.services.impl.cds.AbstractCdsDefinedService;
import com.sap.cds.services.impl.persistence.JdbcPersistenceServiceConfiguration;
import com.sap.cds.services.impl.utils.CdsServiceUtils;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.StringUtils;

public class CdsRuntimeBeanDefinitionRegistrar implements ImportBeanDefinitionRegistrar, EnvironmentAware {

	private static final Logger logger = LoggerFactory.getLogger(CdsRuntimeBeanDefinitionRegistrar.class);
	private static boolean springWebAvailable;
	private static boolean springJdbcAvailable;
	private static boolean cdsJdbcAvailable;

	static {
		try {
			springWebAvailable = SimpleUrlHandlerMapping.class.getName() != null;
		} catch (NoClassDefFoundError e) { // NOSONAR
			springWebAvailable = false;
		}

		try {
			springJdbcAvailable = DataSourceTransactionManager.class.getName() != null;
		} catch (NoClassDefFoundError e) { // NOSONAR
			springJdbcAvailable = false;
		}

		try {
			cdsJdbcAvailable = JdbcPersistenceServiceConfiguration.class.getName() != null;
		} catch (NoClassDefFoundError e) { // NOSONAR
			cdsJdbcAvailable = false;
		}
	}

	private Environment environment;

	@Override
	public void registerBeanDefinitions(AnnotationMetadata importingClassMetadata, BeanDefinitionRegistry registry) {
		BootstrapCache bootstrapCache = BootstrapCache.get(environment).ensureServices();
		registerDataSources(registry, bootstrapCache);
		registerServices(registry, bootstrapCache.getCdsRuntime());
		if (!bootstrapCache.getCdsRuntime().getEnvironment().getCdsProperties().getEnvironment().getCommand().isEnabled()) {
			registerAdapters(registry, bootstrapCache.getAdapterFactories());
		}
	}

	private void registerDataSources(BeanDefinitionRegistry registry, BootstrapCache bootstrapCache) {
		CdsProperties properties = bootstrapCache.getCdsRuntime().getEnvironment().getCdsProperties();
		// only configure if no spring datasource is configured, otherwise the configuration of spring.datasource would be ignored
		String springDataSourceUrl = environment.getProperty("spring.datasource.url");
		if(springDataSourceUrl != null) {
			// check for an embedded SQLite database
			if(springDataSourceUrl.contains(":sqlite:") && springDataSourceUrl.contains(":memory:")) {
				logger.debug("Determined DataSource as embedded based on 'spring.datasource.url' configuration.");
				properties.getDataSource().setEmbedded(true);
			}
			logger.info("Found 'spring.datasource.url' configuration: Auto-configuration of DataSource beans is disabled.");
			return;
		}

		if(!properties.getDataSource().getAutoConfig().isEnabled()) {
			logger.info("Auto-configuration of DataSource beans is explicitly disabled.");
			return;
		}

		Map<String, AbstractBeanDefinition> dataSourceBeanDefinitions = new LinkedHashMap<>();
		// create datasources based on descriptors available in the environment
		for(DataSourceDescriptor descriptor : bootstrapCache.getDataSourceDescriptors()) {
			BeanDefinitionBuilder builder = BeanDefinitionBuilder.genericBeanDefinition(DataSourceDescriptorBeanFactory.class);
			builder.addConstructorArgValue(descriptor.getName());
			builder.setPrimary(descriptor.getName().equals(properties.getDataSource().getBinding()));
			dataSourceBeanDefinitions.put(descriptor.getName(), builder.getBeanDefinition());
		}

		// create datasources provided by datasource providers
		Map<String, DataSource> dataSources = bootstrapCache.getDataSources();
		for(Map.Entry<String, DataSource> entry : dataSources.entrySet()) {
			BeanDefinitionBuilder builder = BeanDefinitionBuilder.genericBeanDefinition(DataSourceBeanFactory.class);
			builder.addConstructorArgValue(entry.getKey());
			builder.addConstructorArgValue(entry.getValue().getClass());
			builder.setPrimary(entry.getKey().equals(properties.getDataSource().getBinding()));
			AbstractBeanDefinition beanDefinition = builder.getBeanDefinition();
			beanDefinition.setAttribute(FactoryBean.OBJECT_TYPE_ATTRIBUTE, entry.getValue().getClass());
			dataSourceBeanDefinitions.put(entry.getKey(), beanDefinition);
		}

		Set<String> createdTxMgrs = new HashSet<>();
		Map<String, PersistenceServiceConfig> serviceConfigs = properties.getPersistence().getServices();
		for (PersistenceServiceConfig serviceConfig : serviceConfigs.values()) {
			String binding = StringUtils.isEmpty(serviceConfig.getBinding()) ? serviceConfig.getName() : serviceConfig.getBinding();
			AbstractBeanDefinition definition = dataSourceBeanDefinitions.remove(binding);
			boolean isPrimary = definition != null ? definition.isPrimary() : false;

			if (serviceConfig.isEnabled()) {
				String dataSourceName = serviceConfig.getDataSource();
				if (StringUtils.isEmpty(dataSourceName)) {
					dataSourceName = "ds-" + binding;
					if (definition != null) {
						registry.registerBeanDefinition(dataSourceName, definition);
						logger.info("Registered {}DataSource '{}'", isPrimaryLiteral(isPrimary), dataSourceName);
					}
				}
				String txMgrName = serviceConfig.getTransactionManager();
				if (StringUtils.isEmpty(txMgrName)) {
					txMgrName = "tx-" + binding;
					if (createdTxMgrs.add(txMgrName)) {
						registerTransactionManager(registry, txMgrName, dataSourceName, isPrimary);
					}
				}

				if (!isPrimary) {
					registerPersistenceService(registry, serviceConfig.getName(), dataSourceName, txMgrName);
				}
			}
		}

		int count = dataSourceBeanDefinitions.size();
		for(Map.Entry<String, AbstractBeanDefinition> definition : dataSourceBeanDefinitions.entrySet()) {
			String binding = definition.getKey();
			String dataSourceName = "ds-" + binding;
			String txMgrName = "tx-" + binding;
			boolean isPrimary = definition.getValue().isPrimary();
			registry.registerBeanDefinition(dataSourceName, definition.getValue());
			logger.info("Registered {}DataSource '{}'", isPrimaryLiteral(isPrimary), dataSourceName);
			registerTransactionManager(registry, txMgrName, dataSourceName, isPrimary);
			if (!isPrimary && count > 1) {
				registerPersistenceService(registry, binding, dataSourceName, txMgrName);
			}
		}

	}

	private void registerTransactionManager(BeanDefinitionRegistry registry, String txMgrName, String dataSourceName, boolean isPrimary) {
		if (springJdbcAvailable) {
			BeanDefinitionBuilder builder = BeanDefinitionBuilder.genericBeanDefinition(DataSourceTransactionManagerBeanFactory.class);
			builder.addConstructorArgValue(dataSourceName);
			builder.setPrimary(isPrimary);
			registry.registerBeanDefinition(txMgrName, builder.getBeanDefinition());
			logger.info("Registered {}TransactionManager '{}'", isPrimaryLiteral(isPrimary), txMgrName);
		}
	}

	private void registerPersistenceService(BeanDefinitionRegistry registry, String binding, String dataSourceName, String txMgrName) {
		if (springJdbcAvailable && cdsJdbcAvailable) {
			BeanDefinitionBuilder persistenceServiceBuilder = BeanDefinitionBuilder.genericBeanDefinition(JdbcPersistenceServiceBeanFactory.class);
			persistenceServiceBuilder.addConstructorArgValue(binding);
			persistenceServiceBuilder.addConstructorArgValue(dataSourceName);
			persistenceServiceBuilder.addConstructorArgValue(txMgrName);
			registry.registerBeanDefinition(binding, persistenceServiceBuilder.getBeanDefinition());
			logger.debug("Registered JdbcPersistenceServiceConfiguration '{}'", binding);
		}
	}

	private void registerServices(BeanDefinitionRegistry registry, CdsRuntime runtime) {
		runtime.getServiceCatalog().getServices().forEach((service) -> {
			Class<?> serviceClass = CdsServiceUtils.getServiceType(service);
			BeanDefinitionBuilder builder = BeanDefinitionBuilder.genericBeanDefinition(ServiceBeanFactory.class);
			builder.addConstructorArgValue(service.getName());
			builder.addConstructorArgValue(serviceClass);
			if (service instanceof ApplicationService || service instanceof RemoteService) {
				// mark the service whose name equals its CDS definition as primary
				String cdsName = AbstractCdsDefinedService.downcast(service).getDefinition().getQualifiedName();
				builder.setPrimary(service.getName().equals(cdsName));
			}
			AbstractBeanDefinition beanDefinition = builder.getBeanDefinition();
			beanDefinition.setAttribute(FactoryBean.OBJECT_TYPE_ATTRIBUTE, serviceClass);
			registry.registerBeanDefinition(service.getName(), beanDefinition);
		});
	}

	private void registerAdapters(BeanDefinitionRegistry registry, Map<String, AdapterFactory> adapterFactories) {
		for(Map.Entry<String, AdapterFactory> entry : adapterFactories.entrySet()) {
			AdapterFactory factory = entry.getValue();
			// skip initialization of servlet adapters, if spring-web is not available
			if(factory instanceof ServletAdapterFactory && !springWebAvailable) {
				continue;
			}

			if(factory.isEnabled()) {
				// create a bean for the adapter
				BeanDefinitionBuilder builder = BeanDefinitionBuilder.genericBeanDefinition(AdapterBeanFactory.class);
				builder.addConstructorArgValue(entry.getKey());
				registry.registerBeanDefinition(entry.getKey(), builder.getBeanDefinition());
			}
		}
	}

	private String isPrimaryLiteral(boolean isPrimary) {
		return isPrimary ? "primary " : "";
	}

	@Override
	public void setEnvironment(Environment environment) {
		this.environment = environment;
	}

}
