/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.framework.spring.actuator;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.function.Predicate;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.actuate.endpoint.annotation.Endpoint;
import org.springframework.boot.actuate.endpoint.annotation.ReadOperation;
import org.springframework.boot.actuate.endpoint.annotation.Selector;
import org.springframework.http.HttpStatus;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.server.ResponseStatusException;

import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.runtime.ExtendedServiceLoader;
import com.sap.cds.services.utils.StringUtils;
import com.sap.cds.services.utils.info.CdsInfo;
import com.sap.cds.services.utils.info.CdsInfo.Details;

@Endpoint(id = "cds")
public class CdsActuatorEndpoint {

	private static final Logger logger = LoggerFactory.getLogger(CdsActuatorEndpoint.class);

	private final Details defaultDetails;
	private final List<CdsInfo> cdsInfoList = new ArrayList<>();

	public CdsActuatorEndpoint(CdsRuntime runtime, Details defaultDetails) {
		ExtendedServiceLoader.loadAll(CdsInfo.class, runtime).forEachRemaining(cdsInfoList::add);
		this.defaultDetails = defaultDetails;
	}

	@ReadOperation
	public Map<String, Object> info(@RequestParam @org.springframework.lang.Nullable String details) {
		return createCdsInfo(i -> true, gainDetails(details)); // NOSONAR
	}

	@ReadOperation
	public Map<String, Object> infoEntry(@Selector String info, @RequestParam @org.springframework.lang.Nullable String details) {
		return createCdsInfo(i -> i.name().equalsIgnoreCase(info), gainDetails(details)); // NOSONAR
	}

	private Details gainDetails(String details) {
		if (!StringUtils.isEmpty(details)) {
			try {
				return Details.valueOf(details.toUpperCase(Locale.ENGLISH));
			} catch(Exception e) { // NOSONAR
				throw new ResponseStatusException(HttpStatus.BAD_REQUEST, "Invalid detail parameter '" + details + "'");
			}
		}
		return defaultDetails;
	}

	private Map<String, Object> createCdsInfo(Predicate<CdsInfo> filter, Details details) {
		Map<String, Object> cdsInfo = new LinkedHashMap<>();

		cdsInfoList.stream().filter(filter).forEach(section -> {
			try {
				cdsInfo.put(section.name(), section.info(details));
			} catch (Exception e) {
				logger.error("Actuator section '{}' failed to gain info", section.name(), e);
			}
		});

		return cdsInfo;
	}
}
