/*********************************************************************
 * (C) 2024 SAP SE or an SAP affiliate company. All rights reserved. *
 *********************************************************************/
package com.sap.cds.framework.spring.actuator;

import java.time.Duration;
import java.time.Instant;

import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.actuate.health.Health;
import org.springframework.boot.actuate.health.HealthIndicator;

import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cloud.sdk.cloudplatform.connectivity.DestinationAccessor;
import com.sap.cloud.sdk.cloudplatform.connectivity.HttpClientAccessor;
import com.sap.cloud.sdk.cloudplatform.resilience.ResilienceConfiguration;
import com.sap.cloud.sdk.cloudplatform.resilience.ResilienceConfiguration.TimeLimiterConfiguration;
import com.sap.cloud.sdk.cloudplatform.resilience.ResilienceDecorator;

class DestinationBasedHealthIndicator implements HealthIndicator {

	private static final Logger logger = LoggerFactory.getLogger(DestinationBasedHealthIndicator.class);
	private static final Duration TIMEOUT_DURATION = Duration.ofSeconds(1);
	private static final Duration STATUS_TIME_TO_LIVE = Duration.ofSeconds(10);

	private final CdsRuntime runtime;
	private final String destinationName;
	private final String healthCheckName;
	private HttpClient httpClient;
	private Health lastHealth;
	private Instant lastCheckedAt;
	private final ResilienceConfiguration resilienceConfiguration;

	DestinationBasedHealthIndicator(CdsRuntime runtime, String destinationName, String healthCheckName) {
		this.runtime = runtime;
		this.destinationName = destinationName;
		this.healthCheckName = healthCheckName;
		this.resilienceConfiguration = ResilienceConfiguration.of(destinationName + ".health")
				.timeLimiterConfiguration(TimeLimiterConfiguration.of(TIMEOUT_DURATION));
	}

	@Override
	public Health health() {
		// NB: Destination created outside of health indicator, but might not be available in the constructor
		if (httpClient == null) {
			httpClient = HttpClientAccessor.getHttpClient(DestinationAccessor.getDestination(destinationName).asHttp());
		}

		Instant now = Instant.now();
		if (lastCheckedAt == null || lastCheckedAt.plus(STATUS_TIME_TO_LIVE).isBefore(now)) {
			lastCheckedAt = now;

			lastHealth = runtime.requestContext().systemUserProvider().run(r -> {
				try {
					return ResilienceDecorator.executeCallable(() -> {
						HttpResponse response = httpClient.execute(new HttpGet("/health"));
						// No response expected from the sidecar
						EntityUtils.consumeQuietly(response.getEntity());
						int statusCode = response.getStatusLine().getStatusCode();
						if (HttpStatus.SC_OK == statusCode) {
							return Health.up().build();
						}
						logger.debug("Health check {} failed with status code {}", healthCheckName, statusCode);
						return Health.down().build();
					}, resilienceConfiguration);
				} catch (Exception e) {
					logger.debug("Health check {} failed with an exception", healthCheckName, e);
					return Health.down().build();
				}
			});
		}
		return lastHealth;
	}
}
