/*
 * © 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.framework.spring.config.auth;

import static com.sap.cds.framework.spring.config.auth.CdsHttpSecurityConfigurerConfig.matcher;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.autoconfigure.AutoConfiguration;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.autoconfigure.condition.ConditionalOnWebApplication;
import org.springframework.context.annotation.Bean;
import org.springframework.core.annotation.Order;
import org.springframework.core.env.Environment;
import org.springframework.security.config.annotation.web.builders.HttpSecurity;
import org.springframework.security.config.annotation.web.configuration.EnableWebSecurity;
import org.springframework.security.web.SecurityFilterChain;
import org.springframework.security.web.util.matcher.OrRequestMatcher;

/**
 * This class is responsible for configuring the security for the health probes. It is only active
 * if the spring-security is present and the health probes endpoints are enabled.
 */
@AutoConfiguration
@ConditionalOnClass(HttpSecurity.class) // only if spring-security
@ConditionalOnWebApplication
@ConditionalOnProperty(
    prefix = "management.endpoint.health.probes",
    name = "enabled",
    havingValue = "true")
@EnableWebSecurity
@Order(100)
public class HealthProbesSecurityConfig {
  private static final Logger logger = LoggerFactory.getLogger(HealthProbesSecurityConfig.class);

  private final Environment env;

  public HealthProbesSecurityConfig(Environment env) {
    this.env = env;
  }

  @Bean
  public SecurityFilterChain appFilterChain(HttpSecurity http) throws Exception {
    String basePath = this.env.getProperty("management.endpoints.web.base-path", "/actuator");
    String readinessPath = "%s/health/readiness".formatted(basePath);
    String livenessPath = "%s/health/liveness".formatted(basePath);
    logger.debug("Opening endpoints for health probes {} and {}.", readinessPath, livenessPath);
    return http.securityMatcher(new OrRequestMatcher(matcher(readinessPath), matcher(livenessPath)))
        .csrf(c -> c.disable())
        .authorizeHttpRequests(r -> r.anyRequest().permitAll())
        .build();
  }
}
