/*
 * © 2021-2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.framework.spring.config.datasource;

import com.sap.cds.framework.spring.config.runtime.BootstrapCache;
import javax.sql.DataSource;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.BeanFactory;
import org.springframework.beans.factory.BeanFactoryAware;
import org.springframework.beans.factory.FactoryBean;

/** Auto-configures a DataSource in Spring, based on an existing {@link DataSource} */
public class DataSourceBeanFactory implements FactoryBean<DataSource>, BeanFactoryAware {

  private final String dataSourceName;
  private final Class<?> dataSourceClass;
  private BeanFactory beanFactory;
  private DataSource dataSource;

  public DataSourceBeanFactory(String dataSourceName, Class<?> dataSourceClass) {
    this.dataSourceName = dataSourceName;
    this.dataSourceClass = dataSourceClass;
  }

  @Override
  public DataSource getObject() throws Exception {
    if (dataSource == null) {
      BootstrapCache bootstrapCache = beanFactory.getBean(BootstrapCache.class);
      dataSource = bootstrapCache.getDataSources().get(dataSourceName);
    }
    return dataSource;
  }

  @Override
  public Class<?> getObjectType() {
    return dataSourceClass == null ? DataSource.class : dataSourceClass;
  }

  @Override
  public void setBeanFactory(BeanFactory beanFactory) throws BeansException {
    this.beanFactory = beanFactory;
  }
}
