/*
 * © 2022-2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.framework.spring.config.datasource;

import com.sap.cds.framework.spring.config.runtime.SpringJdbcPersistenceServiceConfiguration;
import com.sap.cds.services.impl.persistence.JdbcPersistenceService;
import com.sap.cds.services.impl.persistence.JdbcPersistenceServiceConfiguration;
import com.sap.cds.services.runtime.CdsRuntimeConfigurer;
import javax.sql.DataSource;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.BeanFactory;
import org.springframework.beans.factory.BeanFactoryAware;
import org.springframework.beans.factory.SmartFactoryBean;
import org.springframework.transaction.PlatformTransactionManager;

public class JdbcPersistenceServiceBeanFactory
    implements SmartFactoryBean<JdbcPersistenceService>, BeanFactoryAware {

  private final String serviceName;
  private final String dataSourceName;
  private final String txMgrName;

  private BeanFactory beanFactory;
  private JdbcPersistenceServiceConfiguration configuration;

  public JdbcPersistenceServiceBeanFactory(
      String serviceName, String dataSourceName, String txMgrName) {
    this.serviceName = serviceName;
    this.dataSourceName = dataSourceName;
    this.txMgrName = txMgrName;
  }

  @Override
  public JdbcPersistenceService getObject() throws Exception {
    if (configuration == null) {
      DataSource dataSource = beanFactory.getBean(dataSourceName, DataSource.class);
      PlatformTransactionManager txMgr =
          beanFactory.getBean(txMgrName, PlatformTransactionManager.class);
      configuration =
          SpringJdbcPersistenceServiceConfiguration.createJdbcPersistenceServiceConfiguration(
              serviceName, dataSource, txMgr);
    }
    CdsRuntimeConfigurer configurer = beanFactory.getBean(CdsRuntimeConfigurer.class);
    return configuration.createOrGetService(configurer);
  }

  @Override
  public Class<?> getObjectType() {
    return JdbcPersistenceService.class;
  }

  @Override
  public boolean isEagerInit() {
    return true;
  }

  @Override
  public void setBeanFactory(BeanFactory beanFactory) throws BeansException {
    this.beanFactory = beanFactory;
  }
}
