/*
 * © 2019-2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.framework.spring.config.datasource;

import com.sap.cds.services.transaction.TransactionManager;
import java.util.Stack;
import org.springframework.transaction.PlatformTransactionManager;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.TransactionStatus;
import org.springframework.transaction.interceptor.TransactionAspectSupport;
import org.springframework.transaction.support.DefaultTransactionDefinition;
import org.springframework.transaction.support.TransactionSynchronizationManager;

public class SpringTransactionManager implements TransactionManager {

  private final String name;
  private final PlatformTransactionManager txMgr;
  private final ThreadLocal<Stack<TransactionStatus>> transactions =
      ThreadLocal.withInitial(() -> new Stack<>());

  public SpringTransactionManager(String name, PlatformTransactionManager txMgr) {
    this.name = name;
    this.txMgr = txMgr;
  }

  @Override
  public boolean isActive() {
    // check global transaction flag and not only transactions managed by us
    return TransactionSynchronizationManager.isActualTransactionActive();
  }

  @Override
  public void begin() {
    begin(new DefaultTransactionDefinition(TransactionDefinition.PROPAGATION_REQUIRES_NEW));
  }

  void begin(TransactionDefinition definition) {
    // Transactions started before the ChangeSet scope are OK
    // -> ChangeSetContext defines new "Requires New" boundaries
    // Transactions started within the ChangeSet scope but after this begin are OK
    // -> Transactions take part in opened transaction (Requires)
    // Transactions started within the ChangeSet scope but before this begin are OK
    // -> begin is called by the proxy, if transaction with "Requires" is opened
    // !!! Additional "Requires New" might cut the ChangeSet in parts!
    TransactionStatus tx = null;
    try {
      tx = txMgr.getTransaction(definition);
    } finally {
      // if getTransaction throws, null will be pushed
      transactions.get().push(tx);
    }
  }

  @Override
  public void commit() {
    TransactionStatus tx = transactions.get().pop();
    if (tx != null) {
      txMgr.commit(tx);
    }
  }

  @Override
  public void rollback() {
    TransactionStatus tx = transactions.get().pop();
    if (tx != null) {
      txMgr.rollback(tx);
    }
  }

  @Override
  public boolean isRollbackOnly() {
    TransactionStatus tx = transactions.get().peek();
    if (tx != null) {
      return tx.isRollbackOnly();
    } else {
      // if null was pushed because getTransaction threw an exception, we return true
      return true;
    }
  }

  @Override
  public void setRollbackOnly() {
    // works only for transactions opened by @Transactional annotations
    TransactionAspectSupport.currentTransactionStatus().setRollbackOnly();
  }

  @Override
  public String getName() {
    return name;
  }
}
