/*
 * © 2019-2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.framework.spring.config.runtime;

import com.sap.cds.services.environment.CdsProperties;
import com.sap.cds.services.environment.PropertiesProvider;
import com.sap.cds.services.utils.StringUtils;
import java.util.Map;
import org.springframework.boot.context.properties.bind.BindResult;
import org.springframework.boot.context.properties.bind.Bindable;
import org.springframework.boot.context.properties.bind.Binder;
import org.springframework.core.env.Environment;

/** A {@code PropertiesProvider} exposing access to all Spring properties. */
public class SpringPropertiesProvider implements PropertiesProvider {

  static String CDS_ENVIRONMENT_PRODUCTION_ENABLED = "cds.environment.production.enabled";
  static String CDS_ENVIRONMENT_PRODUCTION_PROFILE = "cds.environment.production.profile";

  private final Environment environment;

  public SpringPropertiesProvider(Environment environment) {
    this.environment = environment;
  }

  @Override
  @SuppressWarnings("unchecked")
  public <T> T bindPropertyClass(String section, Class<T> clazz) {
    Bindable<?> bindable;
    if (clazz.equals(CdsProperties.class) && isProduction()) {
      // TODO generalize this also for other PropertiesProvider
      bindable = Bindable.ofInstance(withProductionDefaults());
    } else {
      bindable = Bindable.of(clazz);
    }

    BindResult<?> bindResult = Binder.get(environment).bind(section, bindable);
    if (bindResult.isBound()) {
      return (T) bindResult.get();
    } else if (bindable.getValue() != null) {
      return (T) bindable.getValue().get();
    } else {
      try {
        return clazz.getDeclaredConstructor().newInstance();
      } catch (ReflectiveOperationException e) { // NOSONAR
        return null;
      }
    }
  }

  private boolean isProduction() {
    if (environment.containsProperty(CDS_ENVIRONMENT_PRODUCTION_ENABLED)) {
      return environment.getProperty(CDS_ENVIRONMENT_PRODUCTION_ENABLED, Boolean.class);
    }
    String productionProfile = environment.getProperty(CDS_ENVIRONMENT_PRODUCTION_PROFILE, "cloud");
    if (!StringUtils.isEmpty(productionProfile)) {
      return environment.matchesProfiles(productionProfile);
    }
    return false;
  }

  private CdsProperties withProductionDefaults() {
    CdsProperties cds = new CdsProperties();
    cds.getEnvironment().getProduction().setEnabled(true);
    cds.getIndexPage().setEnabled(false);
    cds.getSecurity().getMock().setEnabled(false);
    cds.getSecurity().getAuthentication().getInternalUserAccess().setEnabled(false);
    return cds;
  }

  @Override
  public String getProperty(String key, String defaultValue) {
    return environment.getProperty(key, defaultValue);
  }

  @Override
  public <T> T getProperty(String key, Class<T> clazz, T defaultValue) {
    if (Map.class.isAssignableFrom(clazz)) {
      return bindPropertyClass(key, clazz);
    }
    return environment.getProperty(key, clazz, defaultValue);
  }
}
