/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.integration.cloudsdk.handler;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Stream;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.sap.cds.services.application.ApplicationLifecycleService;
import com.sap.cds.services.application.ApplicationPreparedEventContext;
import com.sap.cds.services.environment.ServiceBinding;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.Before;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.utils.environment.VcapServicesParser;
import com.sap.cloud.sdk.cloudplatform.CloudPlatformAccessor;

@ServiceName(ApplicationLifecycleService.DEFAULT_NAME)
public class CloudplatformEnvironmentHandler implements EventHandler {

	private static final Logger logger = LoggerFactory.getLogger(CloudplatformEnvironmentHandler.class);

	@Before
	public void registerEnvironmentHandler(ApplicationPreparedEventContext context) {
		CloudPlatformAccessor.tryGetCloudPlatform().andThen((platform) -> {
			platform.setEnvironmentVariableReader(new CdsEnvironmentVariableReader(context.getCdsRuntime().getEnvironment().getServiceBindings()));
		});
	}

	static class CdsEnvironmentVariableReader implements Function<String, String> {

		private String vcapServices = null;

		public CdsEnvironmentVariableReader(Stream<ServiceBinding> bindings) {
			Map<String, List<Map<String, Object>>> serviceToBinding = new HashMap<>();
			bindings.forEach(binding -> {
				List<Map<String, Object>> service = serviceToBinding.get(binding.getService());
				if(service == null) {
					service = new ArrayList<>();
					serviceToBinding.put(binding.getService(), service);
				}
				service.add(asMap(binding));
			});

			try {
				this.vcapServices = new ObjectMapper().writeValueAsString(serviceToBinding);
			} catch (JsonProcessingException e) { // NOSONAR
				logger.warn("Could not serialize service bindings to JSON", e);
			}
		}

		private Map<String, Object> asMap(ServiceBinding binding) {
			Map<String, Object> map = new HashMap<>();
			map.put("name", binding.getName());
			map.put("label", binding.getService());
			map.put("plan", binding.getServicePlan());
			map.put("tags", binding.getTags());
			map.put("credentials", binding.getCredentials());
			return map;
		}

		@Override
		public String apply(String name) {
			// TODO also handle VCAP_APPLICATION?
			if(name.equals(VcapServicesParser.VCAP_SERVICES)) {
				return vcapServices;
			}
			return System.getenv(name);
		}

	}

}
