/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.integration.cloudsdk.handler;

import java.util.concurrent.Callable;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.integration.cloudsdk.decorator.CdsThreadContextDecorator;
import com.sap.cds.services.application.ApplicationLifecycleService;
import com.sap.cds.services.application.ApplicationPreparedEventContext;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.Before;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cloud.sdk.cloudplatform.resilience.ResilienceConfiguration;
import com.sap.cloud.sdk.cloudplatform.resilience.ResilienceDecorator;
import com.sap.cloud.sdk.cloudplatform.thread.DefaultThreadContextDecoratorChain;
import com.sap.cloud.sdk.cloudplatform.thread.ThreadContextExecutorService;
import com.sap.cloud.sdk.frameworks.resilience4j.GenericDecorator;
import com.sap.cloud.sdk.frameworks.resilience4j.Resilience4jDecorationStrategy;

@ServiceName(ApplicationLifecycleService.DEFAULT_NAME)
public class ThreadContextHandler implements EventHandler {

	private static final Logger logger = LoggerFactory.getLogger(ThreadContextHandler.class);
	private static boolean cloudSDK4Available;

	static {
		try {
			cloudSDK4Available = ThreadContextExecutorService.class.getName() != null;
		} catch (NoClassDefFoundError e) { // NOSONAR
			cloudSDK4Available = false;
		}
	}

	@Before
	public void registerDecorators(ApplicationPreparedEventContext context) {
		if(cloudSDK4Available) {
			logger.debug("Found Cloud SDK 4 executor service, adding {}", CdsThreadContextDecorator.class.getSimpleName());
			registerThreadContextDecorator(context);
		} else {
			logger.debug("Could not find Cloud SDK 4 executor service on the classpath. Automated propagation of RequestContext only available for resilience features.");
			registerResilienceThreadContextDecoratorForCompatibility(context);
		}
	}

	private void registerThreadContextDecorator(ApplicationPreparedEventContext context) {
		DefaultThreadContextDecoratorChain.registerDefaultDecorator(new CdsThreadContextDecorator(context.getCdsRuntime()));
	}

	private void registerResilienceThreadContextDecoratorForCompatibility(ApplicationPreparedEventContext context) {
		GenericDecorator customDecorator = new GenericDecorator() {

			private final CdsThreadContextDecorator decorator = new CdsThreadContextDecorator(context.getCdsRuntime());

			@Override
			public <T> Callable<T> decorateCallable(Callable<T> callable, ResilienceConfiguration configuration) {
				return decorator.decorateCallable(callable);
			}

		};
		ResilienceDecorator.setDecorationStrategy(Resilience4jDecorationStrategy.builder().decorator(customDecorator).defaultDecorators().build());
	}
}
