package com.sap.cds.integration.cloudsdk.destination;

import java.util.Map;

import com.sap.cds.services.environment.ServiceBinding;
import com.sap.cloud.sdk.cloudplatform.connectivity.OAuth2DestinationBuilder;
import com.sap.cloud.sdk.cloudplatform.connectivity.OnBehalfOf;
import com.sap.cloud.sdk.cloudplatform.connectivity.HttpDestination;
import com.sap.cloud.security.config.ClientCredentials;
import com.sap.cloud.security.config.ClientCertificate;
import com.sap.cloud.security.config.ClientIdentity;

public class DestinationResolver {

	private DestinationResolver() {
	}

	/**
	 * Returns an instance of {@link HttpDestination} for a given {@link ServiceBinding}. The method will assume
	 * that the binding is based on xsuaa and parse accordingly.
	 *
	 * @param endpointUrl Url of the service
	 * @param binding {@link ServiceBinding} to parse
	 * @param onBehalfOf Defines a specific {@link OnBehalfOf} value which determines authentication type and target tenant of the destination
	 * @return Created {@link HttpDestination} for the given service.
	 */
	public static HttpDestination getDestinationForXsuaaBasedServiceBinding(String endpointUrl, ServiceBinding binding,
			OnBehalfOf onBehalfOf) {

		Map<String, Object> credentials = binding.getCredentials();
		Map<String, Object> uaa = (Map<String, Object>) credentials.get("uaa");

		ClientIdentity client;
		// we don´t need the subdomain of the subscriber
		// cloud sdk/xsuaa will automatically propagate tenant as x-zid header
		String tokenUrl;
		if("x509".equals(uaa.get("credential-type"))) {
			client = new ClientCertificate((String)uaa.get("certificate"), (String) uaa.get("key"), (String) uaa.get("clientid"));
			tokenUrl = (String) uaa.get("certurl");
		} else {
			client = new ClientCredentials((String) uaa.get("clientid"),
					(String) uaa.get("clientsecret"));
			tokenUrl = (String) uaa.get("url");
		}

		return OAuth2DestinationBuilder
			.forTargetUrl(endpointUrl)
			.withTokenEndpoint(tokenUrl)
			.withClient(client, onBehalfOf)
			.build();
	}
}
