/**************************************************************************
 * (C) 2019-2020 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.integration.cloudsdk.facade;

import java.net.URI;

import com.sap.cds.services.request.RequestContext;
import com.sap.cds.services.request.UserInfo;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cloud.sdk.cloudplatform.tenant.DefaultTenant;
import com.sap.cloud.sdk.cloudplatform.tenant.Tenant;
import com.sap.cloud.sdk.cloudplatform.tenant.TenantFacade;
import com.sap.cloud.sdk.cloudplatform.tenant.TenantWithSubdomain;

import io.vavr.control.Try;

public class CdsTenantFacade implements TenantFacade {

	private final CdsRuntime runtime;

	public CdsTenantFacade(CdsRuntime runtime) {
		this.runtime = runtime;
	}

	@Override
	public Try<Tenant> tryGetCurrentTenant() {
		UserInfo userInfo = RequestContext.getCurrent(runtime).getUserInfo();
		String tenantId = userInfo.getTenant();
		if(tenantId == null) {
			return Try.failure(new ErrorStatusException(CdsErrorStatuses.TENANT_CONTEXT_MISSING));
		}

		String subdomain = XsuaaTenant.getSubdomain(userInfo);
		Tenant tenant = subdomain != null ? new XsuaaTenant(tenantId, subdomain) : new DefaultTenant(tenantId);
		return Try.success(tenant);
	}

	private static class XsuaaTenant extends DefaultTenant implements TenantWithSubdomain {

		private static String getSubdomain(UserInfo userInfo) {
			Object issuer = userInfo.getAdditionalAttribute("iss");
			if(issuer != null) {
				try {
					String host = URI.create(issuer.toString()).getHost();
					if(host != null) {
						int subdomainEnd = host.indexOf('.');
						if(subdomainEnd > 0) {
							return host.substring(0, subdomainEnd);
						}
					}
				} catch (IllegalArgumentException e) { // NOSONAR
					// ignore
				}
			}
			return null;
		}

		private final String subdomain;

		public XsuaaTenant(String tenant, String subdomain) {
			super(tenant);
			this.subdomain = subdomain;
		}

		@Override
		public String getSubdomain() {
			return subdomain;
		}

	}

}
