/**************************************************************************
 * (C) 2019-2020 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.integration.cloudsdk.handler;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.sap.cds.feature.platform.PlatformEnvironment;
import com.sap.cds.feature.platform.ServiceBinding;
import com.sap.cds.feature.platform.util.VcapServicesParser;
import com.sap.cds.services.application.ApplicationLifecycleService;
import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.Before;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cloud.sdk.cloudplatform.CloudPlatformAccessor;

@ServiceName(ApplicationLifecycleService.DEFAULT_NAME)
public class CloudplatformEnvironmentHandler implements EventHandler {

	private static final ObjectMapper mapper = new ObjectMapper();
	private static final Logger logger = LoggerFactory.getLogger(CloudplatformEnvironmentHandler.class);

	@Before(event = ApplicationLifecycleService.EVENT_APPLICATION_LIFECYCLE_PREPARED)
	public void registerEnvironmentHandler() {
		CloudPlatformAccessor.tryGetCloudPlatform().andThen((platform) -> {
			platform.setEnvironmentVariableReader(new CdsEnvironmentVariableReader(PlatformEnvironment.INSTANCE));
		});
	}

	static class CdsEnvironmentVariableReader implements Function<String, String> {

		private String vcapServices = null;

		public CdsEnvironmentVariableReader(PlatformEnvironment platformEnvironment) {
			List<ServiceBinding> bindings = platformEnvironment.getServiceBindings().collect(Collectors.toList());
			Map<String, List<ServiceBinding>> serviceToBinding = new HashMap<>();
			for(ServiceBinding binding : bindings) {
				List<ServiceBinding> service = serviceToBinding.get(binding.getService());
				if(service == null) {
					service = new ArrayList<>();
					serviceToBinding.put(binding.getService(), service);
				}
				service.add(binding);
			}

			try {
				this.vcapServices = mapper.writeValueAsString(serviceToBinding);
			} catch (JsonProcessingException e) { // NOSONAR
				logger.warn("Could not serialize service bindings to JSON", e);
			}
		}

		@Override
		public String apply(String name) {
			// TODO also handle VCAP_APPLICATION?
			if(name.equals(VcapServicesParser.VCAP_SERVICES)) {
				return vcapServices;
			}
			return System.getenv(name);
		}

	}

}
