/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.integration.cloudsdk.rest.client;

import java.io.IOException;
import java.time.Duration;
import java.util.Map;

import org.apache.commons.lang3.exception.ExceptionUtils;
import org.apache.http.HttpRequest;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpDelete;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.sap.cds.integration.cloudsdk.destination.HttpClientProvider;
import com.sap.cds.services.environment.CdsProperties.ConnectionPool;
import com.sap.cloud.sdk.cloudplatform.connectivity.ServiceBindingDestinationLoader;
import com.sap.cloud.sdk.cloudplatform.connectivity.ServiceBindingDestinationOptions;
import com.sap.cloud.sdk.cloudplatform.resilience.ResilienceRuntimeException;
import com.sap.cloud.sdk.cloudplatform.security.exception.TokenRequestDeniedException;
import com.sap.cloud.sdk.cloudplatform.security.exception.TokenRequestFailedException;

/**
 * Implementation of a simple JSON REST client.
 */
public class JsonRestClient {

	private static final Logger logger = LoggerFactory.getLogger(JsonRestClient.class);

	private static final String JSON_CONTENT	= "application/json";
	private static final String CONTENT_TYPE	= "content-type";

	protected final ObjectMapper mapper = new ObjectMapper();
	private final HttpClientProvider clientProvider;

	/**
	 * Creates the json rest client.
	 *
	 * @param options the {@link ServiceBindingDestinationOptions}
	 */
	public JsonRestClient(ServiceBindingDestinationOptions options) {
		this(options, new ConnectionPool(Duration.ofSeconds(60), 2, 20));
	}

	/**
	 * Creates the json rest client.
	 *
	 * @param options the {@link ServiceBindingDestinationOptions}
	 * @param connectionPool the {@link ConnectionPool} settings
	 */
	public JsonRestClient(ServiceBindingDestinationOptions options, ConnectionPool connectionPool) {
		// passing runtime as null is okay, given the assumption that requests are always executed within an existing request context
		this.clientProvider = new HttpClientProvider(ServiceBindingDestinationLoader.defaultLoaderChain().getDestination(options), connectionPool, null);
	}

	/**
	 * Performs a HTTP GET request and returns the JSON response.
	 *
	 * @param path destination URL path
	 *
	 * @return JSON response
	 *
	 * @throws IOException in case of a problem or the connection was aborted
	 * @throws JsonRestClientResponseException in case when the response HTTP is not 2xx
	 */
	public JsonNode getRequest(String path) throws IOException {
		HttpGet get = new HttpGet(path);
		try {
			return handleJsonResponse(handleRequest(get));
		} finally {
			get.releaseConnection();
		}
	}

	/**
	 * Performs a HTTP GET request and returns the response code only.
	 *
	 * @param path destination URL path
	 *
	 * @return response code
	 *
	 * @throws IOException in case of a problem or the connection was aborted
	 */
	public int getRequestWithOnlyResponseCode(String path) throws IOException {
		HttpGet get = new HttpGet(path);
		try (CloseableHttpResponse response = handleRequest(get)) {
			return response.getStatusLine().getStatusCode();
		} finally {
			get.releaseConnection();
		}
	}

	/**
	 * Performs a HTTP DELETE request and returns the JSON response.
	 *
	 * @param path destination URL path
	 *
	 * @return JSON response
	 *
	 * @throws IOException in case of a problem or the connection was aborted
	 * @throws JsonRestClientResponseException in case when the response HTTP is not 2xx
	 */
	public JsonNode deleteRequest(String path) throws IOException {
		HttpDelete del = new HttpDelete(path);
		try {
			return handleJsonResponse(handleRequest(del));
		} finally {
			del.releaseConnection();
		}
	}

	/**
	 * Performs a HTTP PUT request and returns the JSON response.
	 *
	 * @param path destination URL path
	 * @param data JSON request body
	 *
	 * @return JSON response
	 *
	 * @throws IOException in case of a problem or the connection was aborted
	 * @throws JsonRestClientResponseException in case when the response HTTP is not 2xx
	 */
	public JsonNode putRequest(String path, JsonNode data) throws IOException {
		HttpPut put = new HttpPut(path);
		if (data != null) {
			put.setEntity(new StringEntity(mapper.writer().writeValueAsString(data), ContentType.APPLICATION_JSON));
		}
		try {
			return handleJsonResponse(handleRequest(put));
		} finally {
			put.releaseConnection();
		}
	}

	/**
	 * Performs a HTTP POST request and returns the JSON response.
	 *
	 * @param path destination URL path
	 * @param data JSON request body
	 *
	 * @return JSON response
	 *
	 * @throws IOException in case of a problem or the connection was aborted
	 * @throws JsonRestClientResponseException in case when the response HTTP is not 2xx
	 */
	public JsonNode postRequest(String path, JsonNode data) throws IOException {
		String strData = mapper.writer().writeValueAsString(data);
		return postRequest(path, strData, null);
	}

	/**
	 * Performs a HTTP POST request and returns the JSON response.
	 *
	 * @param path destination URL path
	 * @param data plain string request body
	 * @param headers additional request headers
	 *
	 * @return JSON response
	 *
	 * @throws IOException in case of a problem or the connection was aborted
	 * @throws JsonRestClientResponseException in case when the response HTTP is not 2xx
	 */
	public JsonNode postRequest(String path, String data, Map<String, String> headers) throws IOException {
		HttpPost post = new HttpPost(path);

		if (headers != null) {
			headers.forEach(post::setHeader);
		}

		if (data != null) {
			post.setEntity(new StringEntity(data, ContentType.APPLICATION_JSON));
		}
		try {
			return handleJsonResponse(handleRequest(post));
		} finally {
			post.releaseConnection();
		}
	}

	private CloseableHttpResponse handleRequest(HttpRequestBase request) throws IOException {
		setHeaders(request);
		try {
			return ((CloseableHttpClient) clientProvider.get()).execute(request);
		} catch (TokenRequestDeniedException | TokenRequestFailedException | ResilienceRuntimeException e) {
			if (ExceptionUtils.indexOfThrowable(e, TokenRequestDeniedException.class) >= 0) {
				throw new JsonRestClientResponseException(401, "Authentication invalid: " + e.getMessage());
			} else if (ExceptionUtils.indexOfThrowable(e, TokenRequestFailedException.class) >= 0) {
				throw new JsonRestClientResponseException(401, "Authentication failed: " + e.getMessage());
			}
			throw e;
		}
	}


	/**
	 * Sets custom headers of the HTTP request.
	 *
	 * @param request HTTP request
	 */
	private void setHeaders(HttpRequest request) {

		logger.debug("Creating {} request to {}", request.getRequestLine().getMethod(), request.getRequestLine().getUri());

		request.removeHeaders(CONTENT_TYPE);
		request.addHeader(CONTENT_TYPE, JSON_CONTENT);
	}


	/**
	 * Handles HTTP response and parses the JSON content.
	 *
	 */
	private JsonNode handleJsonResponse(CloseableHttpResponse response) throws IOException {
		try (CloseableHttpResponse resp = response){
			int code = resp.getStatusLine().getStatusCode();

			logger.debug("Responded with status code '{}'", code);

			if (code >= 200 && code <= 207) {
				String contentType = JSON_CONTENT;
				if (resp.getEntity() != null) {
					if (resp.getEntity().getContentType() != null) {
						contentType = resp.getEntity().getContentType().getValue();
					}
					if (contentType.contains(JSON_CONTENT)) {
						String jsonData = EntityUtils.toString(resp.getEntity());
						return mapper.readValue(jsonData, JsonNode.class);
					} else {
						throw new IOException("Unexpected response format: Expected JSON but found '" + contentType + "'");
					}
				} else {
					return mapper.readValue("{}", JsonNode.class);
				}
			} else {
				String reason = resp.getStatusLine().getReasonPhrase();
				throw new JsonRestClientResponseException(code, "Unexpected request HTTP response (" + code + ") " + reason);
			}
		}
	}
}
