/*
 * © 2023-2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.integration.cloudsdk.destination;

import com.sap.cds.services.environment.CdsProperties.ConnectionPool;
import com.sap.cds.services.request.RequestContext;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.TenantAwareCache;
import com.sap.cloud.sdk.cloudplatform.connectivity.DefaultHttpClientFactory;
import com.sap.cloud.sdk.cloudplatform.connectivity.HttpDestination;
import java.util.function.Supplier;
import org.apache.http.client.HttpClient;

public class HttpClientProvider implements Supplier<HttpClient> {

  private final TenantAwareCache<HttpClient, Boolean> clientCache;

  public HttpClientProvider(
      HttpDestination destination, ConnectionPool config, CdsRuntime runtime) {
    DefaultHttpClientFactory.DefaultHttpClientFactoryBuilder builder =
        DefaultHttpClientFactory.builder();
    builder.timeoutMilliseconds((int) config.getTimeout().toMillis());
    builder.maxConnectionsPerRoute(config.getMaxConnectionsPerRoute());
    builder.maxConnectionsTotal(config.getMaxConnections());
    DefaultHttpClientFactory factory = builder.build();

    Supplier<String> tenantSupplier;
    if (config.getCombinePools().isEnabled()) {
      tenantSupplier = () -> "global-pool";
    } else {
      tenantSupplier =
          () ->
              RequestContext.isActive()
                  ? RequestContext.getCurrent(runtime).getUserInfo().getTenant()
                  : runtime.getProvidedUserInfo().getTenant();
    }
    this.clientCache =
        TenantAwareCache.create(
            tenantSupplier, () -> factory.createHttpClient(destination), () -> true);
  }

  @Override
  public HttpClient get() {
    return clientCache.findOrCreate();
  }
}
