/*
 * © 2024-2025 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.integration.cloudsdk.destination;

import com.google.common.annotations.VisibleForTesting;
import com.sap.cds.services.environment.CdsProperties.Remote.RemoteServiceConfig;
import com.sap.cds.services.request.RequestContext;
import com.sap.cds.services.request.UserInfo;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.IdentityUtils;
import com.sap.cds.services.utils.StringUtils;
import com.sap.cloud.environment.servicebinding.api.ServiceBinding;
import com.sap.cloud.sdk.cloudplatform.connectivity.BtpServiceOptions;
import com.sap.cloud.sdk.cloudplatform.connectivity.DefaultHttpDestination;
import com.sap.cloud.sdk.cloudplatform.connectivity.DestinationAccessor;
import com.sap.cloud.sdk.cloudplatform.connectivity.DestinationOptions;
import com.sap.cloud.sdk.cloudplatform.connectivity.HttpClientAccessor;
import com.sap.cloud.sdk.cloudplatform.connectivity.HttpDestination;
import com.sap.cloud.sdk.cloudplatform.connectivity.OnBehalfOf;
import com.sap.cloud.sdk.cloudplatform.connectivity.ServiceBindingDestinationLoader;
import com.sap.cloud.sdk.cloudplatform.connectivity.ServiceBindingDestinationOptions;
import com.sap.cloud.sdk.cloudplatform.connectivity.ServiceBindingDestinationOptions.Builder;
import com.sap.cloud.sdk.cloudplatform.connectivity.exception.DestinationAccessException;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Consumer;
import java.util.function.Supplier;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.http.client.HttpClient;

public class RemoteServiceHttpClientProvider {

  private static final String IAS_DEPENDENCY_OPTION = "ias-dependency-name";
  private static final String IAS_DEPENDENCY_DESTINATION_PROPERTY =
      "cloudsdk." + IAS_DEPENDENCY_OPTION;
  private static Map<Pair<Object, OnBehalfOf>, HttpDestination> destinations =
      new ConcurrentHashMap<>();

  private RemoteServiceHttpClientProvider() {
    // hidden
  }

  public static HttpClient getHttpClient(
      RemoteServiceConfig remoteServiceConfig, CdsRuntime runtime) {
    if (StringUtils.isEmpty(remoteServiceConfig.getDestination().getName())) {
      remoteServiceConfig.getDestination().setName(remoteServiceConfig.getName());
    }
    return HttpClientAccessor.getHttpClient(initHttpDestination(remoteServiceConfig, runtime));
  }

  @VisibleForTesting
  static HttpDestination initHttpDestination(
      RemoteServiceConfig remoteServiceConfig, CdsRuntime runtime) {
    if (!remoteServiceConfig.getDestination().getProperties().isEmpty()) {
      return DefaultHttpDestination.fromMap(remoteServiceConfig.getDestination().getProperties())
          .build();
    } else if (!StringUtils.isEmpty(remoteServiceConfig.getBinding().getName())) {
      String name = remoteServiceConfig.getName();
      RemoteServiceConfig.Binding binding = remoteServiceConfig.getBinding();
      String onBehalfOf = binding.getOnBehalfOf();
      Supplier<ServiceBinding> serviceBindingProvider =
          () ->
              runtime
                  .getEnvironment()
                  .getServiceBindings()
                  .filter(b -> b.getName().get().equals(binding.getName()))
                  .findFirst()
                  .orElse(null);

      return retrieveOrInitBindingBasedDestination(
          name,
          onBehalfOf,
          serviceBindingProvider,
          runtime,
          (b) -> {
            String url = binding.getOptions().get("url");
            if (url != null) {
              b.withOption(BtpServiceOptions.AuthenticationServiceOptions.withTargetUri(url));
            }
            String iasDependency = binding.getOptions().get(IAS_DEPENDENCY_OPTION);
            if (iasDependency != null) {
              b.withOption(BtpServiceOptions.IasOptions.withApplicationName(iasDependency));
            }
          });
    } else {
      RemoteServiceConfig.Destination destination = remoteServiceConfig.getDestination();

      DestinationOptions.Builder builder = DestinationOptions.builder();
      if (!StringUtils.isEmpty(destination.getRetrievalStrategy())) {
        builder.parameter(
            "scp.cf.destinationRetrievalStrategy", destination.getRetrievalStrategy());
      }
      if (!StringUtils.isEmpty(destination.getTokenExchangeStrategy())) {
        builder.parameter(
            "scp.cf.destinationTokenExchangeStrategy", destination.getTokenExchangeStrategy());
      }
      HttpDestination httpDestination =
          DestinationAccessor.getLoader()
              .tryGetDestination(destination.getName(), builder.build())
              .getOrElseThrow(
                  failure -> {
                    if (failure instanceof DestinationAccessException daex) {
                      throw daex;
                    } else {
                      throw new DestinationAccessException(
                          "Failed to get destination with name '" + destination.getName() + "'.",
                          failure);
                    }
                  })
              .asHttp();

      if (httpDestination.get(IAS_DEPENDENCY_DESTINATION_PROPERTY).isDefined()) {
        Supplier<ServiceBinding> bindingProvider =
            () ->
                new IdentityUtils(runtime)
                    .getIasServiceBindings().stream()
                        .findFirst()
                        .orElseThrow(
                            () ->
                                new ErrorStatusException(
                                    CdsErrorStatuses.NO_SERVICE_BINDING, "IAS"));
        return retrieveOrInitBindingBasedDestination(
            httpDestination,
            "currentUser",
            bindingProvider,
            runtime,
            (b) -> {
              b.withOption(
                  BtpServiceOptions.IasOptions.withApplicationName(
                      (String) httpDestination.get(IAS_DEPENDENCY_DESTINATION_PROPERTY).get()));
              b.withOption(
                  BtpServiceOptions.AuthenticationServiceOptions.withTargetUri(
                      httpDestination.getUri()));
            });
      }

      return httpDestination;
    }
  }

  private static HttpDestination retrieveOrInitBindingBasedDestination(
      Object key,
      String onBehalfOf,
      Supplier<ServiceBinding> bindingProvider,
      CdsRuntime runtime,
      Consumer<Builder> consumer) {
    // TODO: the following code should ideally be pushed down to the Cloud SDK
    OnBehalfOf effectiveOnBehalf;
    UserInfo userInfo = RequestContext.getCurrent(runtime).getUserInfo();
    if ("currentUser".equals(onBehalfOf)
        && userInfo.isAuthenticated()
        && !userInfo.isSystemUser()) {
      effectiveOnBehalf = OnBehalfOf.NAMED_USER_CURRENT_TENANT;
    } else if ("systemUserProvider".equals(onBehalfOf)) {
      effectiveOnBehalf = OnBehalfOf.TECHNICAL_USER_PROVIDER;
    } else {
      effectiveOnBehalf = OnBehalfOf.TECHNICAL_USER_CURRENT_TENANT;
    }

    Pair<Object, OnBehalfOf> cacheKey = Pair.of(key, effectiveOnBehalf);
    return destinations.computeIfAbsent(
        cacheKey,
        (k) -> {
          Builder builder =
              ServiceBindingDestinationOptions.forService(bindingProvider.get())
                  .onBehalfOf(effectiveOnBehalf);
          consumer.accept(builder);
          return ServiceBindingDestinationLoader.defaultLoaderChain()
              .getDestination(builder.build());
        });
  }
}
