/*
 * © 2020-2025 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.integration.cloudsdk.facade;

import com.sap.cds.services.request.RequestContext;
import com.sap.cds.services.request.UserInfo;
import com.sap.cds.services.runtime.CdsRuntime;
import com.sap.cds.services.utils.CdsErrorStatuses;
import com.sap.cds.services.utils.ErrorStatusException;
import com.sap.cds.services.utils.StringUtils;
import com.sap.cloud.sdk.cloudplatform.tenant.DefaultTenant;
import com.sap.cloud.sdk.cloudplatform.tenant.Tenant;
import com.sap.cloud.sdk.cloudplatform.tenant.TenantFacade;
import io.vavr.control.Try;
import java.net.URI;

public class CdsTenantFacade implements TenantFacade {

  private final CdsRuntime runtime;

  public CdsTenantFacade(CdsRuntime runtime) {
    this.runtime = runtime;
  }

  @Override
  public Try<Tenant> tryGetCurrentTenant() {
    UserInfo userInfo = RequestContext.getCurrent(runtime).getUserInfo();
    String tenantId = userInfo.getTenant();
    if (tenantId == null) {
      return Try.failure(new ErrorStatusException(CdsErrorStatuses.TENANT_CONTEXT_MISSING));
    }

    // Subdomain handling is primarily left for ias use cases.
    // For xsuaa use case, Cloud SDK uses z-id header
    String subdomain = getSubdomain(userInfo);
    Tenant tenant =
        subdomain != null ? new DefaultTenant(tenantId, subdomain) : new DefaultTenant(tenantId);
    return Try.success(tenant);
  }

  private static String getSubdomain(UserInfo userInfo) {

    // "subDomain" attribute is used with priority. It is available in xsuaa and dwc/ias cases
    String subDomain = (String) userInfo.getAdditionalAttribute("subDomain");
    if (!StringUtils.isEmpty(subDomain)) {
      return subDomain;
    }
    // "iss" attribute is used as a fallback. It is available in pure ias cases without dwc
    Object issuer = userInfo.getAdditionalAttribute("iss");
    if (issuer != null) {
      try {
        String host = URI.create(issuer.toString()).getHost();
        if (host != null) {
          int subdomainEnd = host.indexOf('.');
          if (subdomainEnd > 0) {
            return host.substring(0, subdomainEnd);
          }
        }
      } catch (IllegalArgumentException e) { // NOSONAR
        // ignore
      }
    }
    return null;
  }
}
