/*
 * © 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.integration.cloudsdk.handler;

import com.sap.cds.services.handler.EventHandler;
import com.sap.cds.services.handler.annotations.After;
import com.sap.cds.services.handler.annotations.ServiceName;
import com.sap.cds.services.mt.DependenciesEventContext;
import com.sap.cds.services.mt.DeploymentService;
import com.sap.cds.services.utils.environment.ServiceBindingUtils;
import com.sap.cloud.environment.servicebinding.api.ServiceBinding;
import java.util.Collections;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Stream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Destination service related implementation of the multi tenancy subscription handler which is
 * responsible for providing the dependencies to the bound Destination service.
 */
@ServiceName(DeploymentService.DEFAULT_NAME)
public class DestinationServiceMtHandler implements EventHandler {

  private static final Logger logger = LoggerFactory.getLogger(DestinationServiceMtHandler.class);

  private static final String DESTINATION = "destination";
  private String xsappname;
  private String bindingName;

  public DestinationServiceMtHandler(Stream<ServiceBinding> bindings) {
    Optional<ServiceBinding> binding =
        bindings.filter(b -> ServiceBindingUtils.matches(b, DESTINATION)).findFirst();
    if (binding.isPresent()) {
      this.bindingName = binding.get().getName().orElse("");

      Map<String, Object> credentials = binding.get().getCredentials();
      this.xsappname = (String) credentials.get("xsappname");
    }
  }

  @After
  public void addDestinationDependency(DependenciesEventContext context) {
    if (xsappname == null) {
      return;
    }

    logger.debug(
        "Providing Destination service dependency for binding '{}' with xsappname '{}'",
        this.bindingName,
        xsappname);
    context.getResult().add(Collections.singletonMap("xsappname", xsappname));
  }
}
