/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.maven.plugin;

import java.io.File;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;

import org.apache.maven.execution.MavenSession;
import org.apache.maven.model.Plugin;
import org.apache.maven.model.PluginExecution;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecution;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.Component;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.project.MavenProject;
import org.apache.maven.settings.Settings;
import org.codehaus.plexus.util.Scanner;
import org.sonatype.plexus.build.incremental.BuildContext;

/**
 * An abstract base class for all CDS-related Mojos (Maven Old Java Object)
 */
public abstract class AbstractCdsMojo extends AbstractMojo implements CdsMojoLogger {
	/**
	 * The key of this plugin.
	 */
	public static final String PLUGIN_KEY = "com.sap.cds:cds-maven-plugin";

	@Component
	protected BuildContext buildContext;

	/**
	 * The current {@link MojoExecution}. Used to determine if goal is executed from command line or from lifecycle
	 * phase.
	 */
	@Parameter(defaultValue = "${mojoExecution}", required = true, readonly = true)
	protected MojoExecution mojoExecution;

	/**
	 * The current Maven project on which this goal is executed.
	 */
	@Parameter(defaultValue = "${project}", required = true, readonly = true)
	protected MavenProject project;

	/**
	 * The current Maven execution session.
	 */
	@Parameter(defaultValue = "${session}", required = true, readonly = true)
	protected MavenSession session;

	/**
	 * The global Maven settings, stored in <code>~/.m2/settings.xml</code>.
	 */
	@Parameter(defaultValue = "${settings}", required = true, readonly = true)
	protected Settings settings;

	private File reactorBaseDirectory;

	protected AbstractCdsMojo() {
		// to avoid NullPointerExceptions during unit tests
		setPluginContext(new HashMap<>());
	}

	/**
	 * Indicates whether this goal is executed from command line or not.
	 *
	 * @return <code>true</code> if goal is invoked from command line, otherwise <code>false</code>.
	 */
	public boolean isCliExecuted() {
		return this.mojoExecution.getSource() == MojoExecution.Source.CLI;
	}

	@Override
	public void logDebug(String message, Object... args) {
		if (getLog().isDebugEnabled()) {
			getLog().debug(getMessage(message, args));
		}
	}

	@Override
	public void logDebug(Throwable error) {
		if (getLog().isDebugEnabled()) {
			getLog().debug(error);
		}
	}

	@Override
	public void logError(String message, Object... args) {
		if (getLog().isErrorEnabled()) {
			getLog().error(getMessage(message, args));
		}
	}

	@Override
	public void logError(String message, Throwable error, Object... args) {
		if (getLog().isErrorEnabled()) {
			getLog().error(getMessage(message, args), error);
		}
	}

	@Override
	public void logError(Throwable error) {
		if (getLog().isErrorEnabled()) {
			getLog().error(error);
		}
	}

	@Override
	public void logInfo(String message, Object... args) {
		if (getLog().isInfoEnabled()) {
			getLog().info(getMessage(message, args));
		}
	}

	@Override
	public void logWarn(String message, Object... args) {
		if (getLog().isWarnEnabled()) {
			getLog().warn(getMessage(message, args));
		}
	}

	@Override
	public void logWarn(String message, Throwable error, Object... args) {
		if (getLog().isWarnEnabled()) {
			getLog().warn(getMessage(message, args), error);
		}
	}

	@Override
	public void logWarn(Throwable error) {
		if (getLog().isWarnEnabled()) {
			getLog().warn(error);
		}
	}

	/**
	 * Ensures that the goal is executed from command line.
	 *
	 * @throws MojoExecutionException if goal isn't invoked from command line
	 */
	protected void ensureCliExecuted() throws MojoExecutionException {
		if (!isCliExecuted()) {
			throw new MojoExecutionException("This goal can only be executed from command line.");
		}
	}

	protected synchronized File getTopmostProjectDir() {
		if (this.reactorBaseDirectory == null) {
			// start with basedir of the current project
			this.reactorBaseDirectory = this.project.getBasedir();

			// try to find directory of last parent project, that is part of the current source tree
			MavenProject parentProject = this.project.getParent();
			while (parentProject != null && parentProject.getBasedir() != null) {
				this.reactorBaseDirectory = parentProject.getBasedir();
				parentProject = parentProject.getParent();
			}
		}
		return this.reactorBaseDirectory;
	}

	/**
	 * Scans the given directory for files satisfying the given glob pattern.
	 *
	 * @param baseDir  the base directory to start scanning
	 * @param includes an array of glob based file pattern, for example, <code>&#42;&#42;/csn.json</code>
	 * @return a list with files
	 */
	protected List<File> scanDirectory(File baseDir, String... includes) {
		Scanner scanner = this.buildContext.newScanner(baseDir, false);
		scanner.setIncludes(includes);
		if (getLog().isDebugEnabled()) {
			logDebug("Scanning directory %s for %s", baseDir, Arrays.toString(includes));
		}
		scanner.scan();
		String[] includedFiles = scanner.getIncludedFiles();

		if (includedFiles.length == 0) {
			return Collections.emptyList();
		}

		List<File> locatedFiles = new ArrayList<>(includedFiles.length);
		for (String includedFile : includedFiles) {
			File file = new File(baseDir, includedFile);
			locatedFiles.add(file);
			logDebug("Found %s", file);
		}
		return locatedFiles;
	}

	protected void setProperty(String name, File value) {
		// the file path needs to be stored as String
		this.project.getProperties().put(name, value.getAbsolutePath());
	}

	private String getMessage(String message, Object... args) {
		if (message == null) {
			return this.getClass().getSimpleName() + ": null";
		}
		if (args == null || args.length == 0) {
			return this.getClass().getSimpleName() + ": " + message;
		}
		try {
			return this.getClass().getSimpleName() + ": " + String.format(message, args);
		} catch (@SuppressWarnings("unused") Exception ignored) { // NOSONAR
			return this.getClass().getSimpleName() + ": " + message;
		}
	}

	/*
	 * static helpers
	 */
	protected static PluginExecution findGoalExecution(Plugin plugin, String goalName) {
		return plugin.getExecutions().stream() //
				.filter(pluginExec -> pluginExec.getGoals().stream() //
						.anyMatch(goal -> goal.equals(goalName)))
				.findFirst().orElse(null);
	}
}
