/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.maven.plugin.add;

import java.io.File;
import java.io.IOException;

import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.project.MavenProject;

import com.sap.cds.maven.plugin.add.TemplateService.TemplateVariable;
import com.sap.cds.maven.plugin.util.Utils;

/**
 * Add sample Java handler class, unit test, and CDS sample files to the current CAP Java project.<br>
 * Call <code>mvn com.sap.cds:cds-maven-plugin:addSample</code> to add sample classes and CDS files.<br>
 * <br>
 * <strong>Note:</strong> This goal can only be executed from the command line.<br>
 * <br>
 *
 * @since 1.5.0
 */
@Mojo(name = "addSample", aggregator = true, requiresDirectInvocation = true)
public class AddSampleMojo extends AbstractAddMojo {

	static final String CAT_SERVICE_CDS = "cat-service.cds";

	/**
	 * Overwrite existing files if set to <code>true</code>.
	 */
	@Parameter(property = "overwrite", defaultValue = "false")
	private boolean overwrite;

	private final TemplateService templateService = new TemplateService();

	private boolean createIntegrationTestClass;

	@Override
	public void execute() throws MojoExecutionException {
		ensureCliExecuted();

		checkPrerequisites();

		try {
			addSample();
		} catch (IOException e) {
			throw new MojoExecutionException(e.getMessage(), e);
		}
	}

	private void checkPrerequisites() {
		// get ./integration-tests/pom.xml
		File itPomXmlFile = new File(this.project.getBasedir(), AddIntegrationTestMojo.ITEST_MODULE + "/pom.xml");

		this.createIntegrationTestClass = itPomXmlFile.exists();
	}

	private void addSample() throws IOException, MojoExecutionException {
		String applicationPackage = getApplicationPackage();
		String packageName = applicationPackage + ".handlers";

		logDebug("handler package: %s", packageName);
		logDebug("overwrite: %s", this.overwrite);

		MavenProject srvProject = findSrvProject();

		String sourceFolder = srvProject.getBuild().getSourceDirectory();
		logDebug("source folder: %s", sourceFolder);

		String testSourceFolder = srvProject.getBuild().getTestSourceDirectory();
		logDebug("test source folder: %s", testSourceFolder);

		File javaSrcFolder = Utils.createDir(sourceFolder);
		File javaTestSrcFolder = Utils.createDir(testSourceFolder);

		this.templateService.createClassFromTemplate(javaSrcFolder, "/CatalogServiceHandler.java", packageName,
				"CatalogServiceHandler", this.overwrite);
		this.templateService.createClassFromTemplate(javaTestSrcFolder, "/CatalogServiceHandlerTest.java", packageName,
				"CatalogServiceHandlerTest", this.overwrite);

		this.templateService.createFileFromTemplate("/srv/" + CAT_SERVICE_CDS, this.project.getBasedir(),
				this.overwrite);
		this.templateService.createFileFromTemplate("/db/data-model.cds", this.project.getBasedir(), this.overwrite);
		this.templateService.createFileFromTemplate("/db/data/my.bookshop-Books.csv", this.project.getBasedir(),
				this.overwrite);

		// also add integration test class if module already exists
		if (this.createIntegrationTestClass) {
			File itestSourceFolder = new File(this.project.getBasedir(),
					AddIntegrationTestMojo.ITEST_MODULE + "/src/test/java");
			logDebug("integration test source folder: %s", itestSourceFolder);

			String itPackageName = applicationPackage + AddIntegrationTestMojo.IT_PACKAGE;
			logDebug("integration test package: %s", itPackageName);
			TemplateVariable applicationPackagePair = TemplateVariable.of("applicationPackage", applicationPackage);

			getIntegrationTestTemplates().forEach(template -> {
				try {
					this.templateService.createClassFromTemplate(itestSourceFolder, template[0],
							itPackageName, template[1], this.overwrite, applicationPackagePair);
				} catch (IOException e) {
					logError(e);
				}
			});
		}
	}
}
