package com.sap.cds.maven.plugin.build;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.apache.maven.model.Plugin;
import org.apache.maven.model.PluginExecution;
import org.apache.maven.plugin.BuildPluginManager;
import org.apache.maven.plugin.InvalidPluginDescriptorException;
import org.apache.maven.plugin.MojoExecution;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugin.MojoNotFoundException;
import org.apache.maven.plugin.PluginConfigurationException;
import org.apache.maven.plugin.PluginDescriptorParsingException;
import org.apache.maven.plugin.PluginManagerException;
import org.apache.maven.plugin.PluginNotFoundException;
import org.apache.maven.plugin.PluginResolutionException;
import org.apache.maven.plugin.descriptor.MojoDescriptor;
import org.apache.maven.plugins.annotations.Component;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.shared.invoker.CommandLineConfigurationException;
import org.apache.maven.shared.invoker.DefaultInvocationRequest;
import org.apache.maven.shared.invoker.InvocationRequest;
import org.apache.maven.shared.invoker.MavenCommandLineBuilder;
import org.apache.maven.shared.invoker.SystemOutHandler;
import org.apache.maven.shared.utils.cli.Commandline;
import org.apache.maven.shared.utils.cli.StreamPumper;
import org.codehaus.plexus.configuration.PlexusConfiguration;
import org.codehaus.plexus.util.cli.StreamFeeder;
import org.codehaus.plexus.util.xml.Xpp3Dom;
import org.codehaus.plexus.util.xml.Xpp3DomUtils;

import com.sap.cds.maven.plugin.AbstractCdsMojo;

/**
 * An abstract base class for all MOJOs using the Maven invoker or Mojo executor.
 */
abstract class AbstractInvokerMojo extends AbstractCdsMojo {

	/*
	 * Details about supported default properties: \
	 * http://maven.apache.org/components/ref/3-LATEST/maven-model-builder/ \
	 * https://github.com/cko/predefined_maven_properties/blob/master/README.md
	 */

	// Internally used parameter, not configurable in pom.xml
	@Parameter(defaultValue = "${project.basedir}/pom.xml", readonly = true, required = true)
	private File pom;

	// Internally used parameter, not configurable in pom.xml
	@Parameter(defaultValue = "${maven.home}", readonly = true, required = true)
	private File mavenHome;

	/**
	 * The Maven BuildPluginManager component.
	 */
	@Component
	private BuildPluginManager pluginManager;

	protected InvocationRequest createRequest(List<String> goals) {
		InvocationRequest request = new DefaultInvocationRequest();
		request.setPomFile(this.pom);
		request.setGoals(goals);

		// debug mode is enable with cmd line option -X
		request.setDebug(getLog().isDebugEnabled());

		// enable to batch mode to avoid warnings about missing input stream
		request.setBatchMode(true);

		// forward cmd line arguments like -Dcds.generate.codeOutputDirectory=... to invoked goals
		request.setProperties(super.session.getUserProperties());

		return request;
	}

	/**
	 * Creates a new maven command line for given invocation request.
	 *
	 * @param request the {@link InvocationRequest invocation request}, may not <code>null</code>.
	 * @return the newly created {@link Commandline command line} to execute maven in a new process.
	 * @throws MojoExecutionException if creation of command line failed
	 */
	protected Commandline createMvnCmdLine(InvocationRequest request) throws MojoExecutionException {
		MavenCommandLineBuilder builder = new MavenCommandLineBuilder();
		builder.setMavenHome(this.mavenHome);
		try {
			return builder.build(request);
		} catch (CommandLineConfigurationException e) {
			throw new MojoExecutionException(e.getMessage(), e);
		}
	}

	protected void executeMojo(Plugin plugin, String goal, PluginExecution pluginExec) throws MojoExecutionException {
		// get mojo descriptor for given plugin and goal
		MojoDescriptor mojoDescriptor = getMojoDescriptor(plugin, goal);

		executeMojo(mojoDescriptor, pluginExec);
	}

	protected Process executeMaven(InvocationRequest request) throws MojoExecutionException {
		// create a maven command line for given invocation request
		Commandline mvnCmdLine = createMvnCmdLine(request);

		return executeMaven(mvnCmdLine);
	}

	@SuppressWarnings("resource")
	private Process executeMaven(Commandline mvnCmdLine) throws MojoExecutionException {
		// remove quotes again
		String executable = unquote(mvnCmdLine.getExecutable());

		List<String> commandline = new ArrayList<>();
		commandline.add(executable);
		Collections.addAll(commandline, mvnCmdLine.getArguments());

		ProcessBuilder procBuilder = new ProcessBuilder(commandline);
		procBuilder.directory(mvnCmdLine.getWorkingDirectory());

		logInfo("Executing Maven command %s in working directory %s", procBuilder.command(), procBuilder.directory());

		Process mvnProcess;
		try {
			mvnProcess = procBuilder.start();

			StreamPumper pumper = new StreamPumper(mvnProcess.getInputStream(), new SystemOutHandler());
			pumper.start();
			StreamFeeder feeder = new StreamFeeder(System.in, mvnProcess.getOutputStream());
			feeder.start();
			StreamPumper errorPumper = new StreamPumper(mvnProcess.getErrorStream(), new SystemOutHandler());
			errorPumper.start();

			return mvnProcess;
		} catch (IOException e) {
			throw new MojoExecutionException(e.getMessage(), e);
		}
	}

	private MojoDescriptor getMojoDescriptor(Plugin plugin, String goal) throws MojoExecutionException {
		try {
			return this.pluginManager.getMojoDescriptor(plugin, goal, super.project.getRemotePluginRepositories(),
					super.session.getRepositorySession());
		} catch (PluginNotFoundException | PluginResolutionException | PluginDescriptorParsingException
				| MojoNotFoundException | InvalidPluginDescriptorException e) {
			throw new MojoExecutionException(e.getMessage(), e);
		}
	}

	private void executeMojo(MojoDescriptor mojoDescriptor, PluginExecution pluginExec) throws MojoExecutionException {
		MojoExecution exec = createMojoExecution(mojoDescriptor, (Xpp3Dom) pluginExec.getConfiguration());

		try {
			this.pluginManager.executeMojo(super.session, exec);
		} catch (PluginConfigurationException | PluginManagerException | MojoFailureException e) {
			throw new MojoExecutionException(e.getMessage(), e);
		}
	}

	/*
	 * static helpers
	 */
	private static MojoExecution createMojoExecution(MojoDescriptor mojoDescriptor, Xpp3Dom configuration) {
		Xpp3Dom mergedConfig = Xpp3DomUtils.mergeXpp3Dom(configuration,
				toXpp3Dom(mojoDescriptor.getMojoConfiguration()));
		return new MojoExecution(mojoDescriptor, mergedConfig);
	}

	/**
	 * Converts PlexusConfiguration to a Xpp3Dom.
	 *
	 * @param config the PlexusConfiguration. Must not be {@code null}.
	 * @return the Xpp3Dom representation of the PlexusConfiguration
	 */
	private static Xpp3Dom toXpp3Dom(PlexusConfiguration config) {
		Xpp3Dom result = new Xpp3Dom(config.getName());
		result.setValue(config.getValue(null));
		for (String name : config.getAttributeNames()) {
			result.setAttribute(name, config.getAttribute(name));
		}
		for (PlexusConfiguration child : config.getChildren()) {
			result.addChild(toXpp3Dom(child));
		}
		return result;
	}

	private static String unquote(String executable) {
		if ((executable.startsWith("'") && executable.endsWith("'"))
				|| (executable.startsWith("\"") && executable.endsWith("\""))) {
			return executable.substring(1, executable.length() - 1);
		}
		return executable;
	}
}
