package com.sap.cds.maven.plugin.build;

import org.apache.maven.model.Plugin;
import org.apache.maven.model.PluginExecution;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;

/**
 * Performs a cds build with Java class generation by invoking the goals <code>install-node</code>,
 * <code>install-cdsdk</code>, <code>cds</code> and <code>generate</code> of the <code>cds-maven-plugin</code> on
 * current project. Any Java property that is provided on command line with option <code>-D</code> will be forwarded to
 * the invoked goals.<br>
 * Call <code>mvn cds:build</code> or <code>mvn com.sap.cds:cds-maven-plugin:build</code> to perform the cds build.<br>
 * <br>
 * <strong>Note:</strong> This goal can only be executed from the command line.<br>
 * <br>
 *
 * @since 1.17.0
 */
@Mojo(name = "build", defaultPhase = LifecyclePhase.NONE, aggregator = true, requiresDirectInvocation = true)
public class BuildMojo extends AbstractInvokerMojo {

	@Override
	public void execute() throws MojoExecutionException {
		ensureCliExecuted();

		executeBuild();
	}

	protected void executeBuild() throws MojoExecutionException {
		// get cds-maven-plugin from current pom.xml
		Plugin plugin = this.project.getModel().getBuild().getPluginsAsMap().get(PLUGIN_KEY);

		if (plugin != null) {
			executeGoal(plugin, "install-node");
			executeGoal(plugin, "install-cdsdk");
			executeGoal(plugin, "cds");
			executeGoal(plugin, "generate");
		} else {
			logInfo("Plugin %s not used in %s, skipping execution.", PLUGIN_KEY, this.project.getFile());
		}
	}

	private void executeGoal(Plugin plugin, String goal) throws MojoExecutionException {
		PluginExecution pluginExecution = findGoalExecution(plugin, goal);
		if (pluginExecution != null) {
			executeMojo(plugin, goal, pluginExecution);
		} else {
			logDebug("Goal %s of plugin %s not used.", goal, plugin.getKey());
		}
	}
}
