/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.maven.plugin.build;

import static com.sap.cds.maven.plugin.util.Utils.array;
import static java.lang.String.format;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Scanner;

import org.apache.commons.exec.ExecuteException;
import org.apache.maven.model.Plugin;
import org.apache.maven.model.PluginExecution;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.codehaus.plexus.util.StringUtils;
import org.codehaus.plexus.util.xml.Xpp3Dom;

import com.sap.cds.maven.plugin.util.Platform;
import com.sap.cds.maven.plugin.util.Semver;
import com.sap.cds.maven.plugin.util.Utils;

/**
 * Execute CDS commands on the current CAP Java project.<br>
 * Call <code>mvn cds:cds</code> or <code>mvn com.sap.cds:cds-maven-plugin:cds</code> on the command line to execute all
 * configured CDS commands of the project in current directory.<br>
 * <br>
 * Several CDS commands can be configured in one execution block and they’re executed in the specified order. If a
 * command execution fails, the overall goal execution is stopped and the Maven build fails.<br>
 * <br>
 * <strong>Note:</strong>This goal requires an installation of the
 * <a href="https://www.npmjs.com/package/@sap/cds-dk">@sap/cds-dk</a>, either locally or globally. The goal
 * <code>install-cdsdk</code> of this plugin can be used for this.<br>
 * <br>
 *
 * @since 1.7.0
 */
@Mojo(name = "cds", defaultPhase = LifecyclePhase.GENERATE_SOURCES, aggregator = true)
public class CdsMojo extends AbstractNodejsMojo {

	// @TODO: replace it with a minimum version provided by CDS4J
	private static final Semver MIN_CDSDK_VERSION = new Semver("3.0.0");

	@Parameter(defaultValue = "${project.build.directory}/cds/build.log", readonly = true, required = true)
	private File cdsBuildOutputFile;

	/**
	 * A list of commands to be executed. Each command is given to CDS for execution in the specified order.<br>
	 * For example:
	 * <code>build/all --clean, deploy --to sqlite --dry > ${project.basedir}/src/main/resources/schema.sql</code>.
	 */
	@Parameter
	private List<String> commands;

	/**
	 * Determine whether to generate Javadoc for the generated interfaces.
	 *
	 * @since 1.17.0
	 */
	@Parameter(property = "cds.documentation", defaultValue = "true")
	private boolean documentation;

	/**
	 * Skip execution of this goal.
	 */
	@Parameter(property = "cds.cds.skip", defaultValue = "false")
	private boolean skip;

	/**
	 * The working directory to be used during CDS command execution. If it's not specified, the plugin is using the
	 * directory that contains a <STRONG>.cdsrc.json</STRONG> or <STRONG>package.json</STRONG> file. The plugin goes up
	 * the project hierarchy on the filesystem until one of these files is found or the topmost project directory has
	 * been reached.
	 */
	@Parameter
	private File workingDirectory;

	// internally used parameter, not configurable in pom.xml
	@Parameter(defaultValue = PARAM_NPX_EXECUTABLE, readonly = true)
	private File npxExec;

	@Override
	public void execute() throws MojoExecutionException {
		checkCdsdkVersion();

		if (isCliExecuted()) {
			executeViaCli();
		} else {
			executeViaLifecycle();
		}
	}

	/**
	 * Gets called to inject parameter "npxExec".
	 *
	 * @param npxExec a {@link File} pointing to the npx executable.
	 */
	public void setNpxExec(File npxExec) {
		if (npxExec != null && npxExec.canExecute()) {
			this.npxExec = npxExec;
			logInfo("Using npx from previous installation: %s", this.npxExec);
		}
	}

	private void checkCdsdkVersion() throws MojoExecutionException {
		Semver cdsdkSemver = getCdsdkVersion();

		if (cdsdkSemver.compareTo(MIN_CDSDK_VERSION) < 0) {
			throw new MojoExecutionException(format(
					"Minimum required version %s of @sap/cds-dk not available, found version %s. Please update @sap/cds-dk to newer version.",
					MIN_CDSDK_VERSION, cdsdkSemver));
		}
	}

	/**
	 * Executes a CDS command with given arguments in given working directory.
	 *
	 * @param workDir the working directory to use
	 * @param args    the CDS command line arguments
	 * @param output  an optional {@link OutputStream}
	 * @throws ExecuteException if CDS execution failed with an error
	 * @throws IOException      if an I/O error occurred
	 */
	private void executeCds(File workDir, String args, OutputStream output) throws ExecuteException, IOException {
		// split and join to remove ugly whitespace characters like \n,\r,\t, and make command line more robust
		String argLine = StringUtils.join(Utils.splitByWhitespaces(args), " ");

		Utils.prepareDestination(this.cdsBuildOutputFile, false);
		Map<String, String> environment = new HashMap<>();
		environment.put("CDS_BUILD_OUTPUTFILE", this.cdsBuildOutputFile.getAbsolutePath());

		// enable documentation generation if requested
		if (this.documentation) {
			environment.put("CDS_CDSC_DOCS", "true");
		}

		// set MSYSTEM and TERM on windows to empty values to solve issue with wrong environment in git-bash
		// see: https://github.com/npm/npx/issues/78#issuecomment-691102516
		if (Platform.CURRENT.isWindows()) {
			environment.put("MSYSTEM", "");
			environment.put("TERM", "");

			// double quotes need to be escaped on windows platform,
			// otherwise "npx -c cds ..." will fail on project paths containing a space
			argLine = argLine.replace("\"", "\\\"");
		}

		String[] allArgs = array("-c", "cds " + argLine);

		execute(workDir, getNpxExec(), output, environment, null, allArgs);
	}

	private void executeViaCli() throws MojoExecutionException {
		Plugin plugin = this.project.getModel().getBuild().getPluginsAsMap().get(PLUGIN_KEY);
		if (plugin != null) {
			PluginExecution pluginExec = findGoalExecution(plugin, "cds");

			if (pluginExec != null) {
				// get working directory from configuration or fallback to default value
				Xpp3Dom workingDirDom = ((Xpp3Dom) pluginExec.getConfiguration()).getChild("workingDirectory");
				File workingDir = workingDirDom != null ? new File(workingDirDom.getValue()) : getWorkingDirectory();

				Xpp3Dom commandsDom = ((Xpp3Dom) pluginExec.getConfiguration()).getChild("commands");

				for (Xpp3Dom command : commandsDom.getChildren()) {
					try {
						executeCds(workingDir, command.getValue(), null);
					} catch (IOException e) {
						throw new MojoExecutionException(e.getMessage(), e);
					}
				}
			}
		}
	}

	private void executeViaLifecycle() throws MojoExecutionException {
		if (!this.skip) {
			File workDir = getWorkingDirectory();

			try {
				if (this.commands != null && !this.commands.isEmpty()) {
					for (String argLine : this.commands) {
						executeCds(workDir, argLine, null);
					}

					// inform m2e about possible changes in resources directories, for example, edmx directory
					Utils.getResourceDirs(this.project).forEach(resourceDir -> super.buildContext.refresh(resourceDir));
				} else {
					logInfo("No commands configured, nothing to do.");
				}
				// catch everything as m2e doesn't show errors if a thrown exception
			} catch (Exception e) { // NOSONAR
				throw new MojoExecutionException(e.getMessage(), e);
			}
		} else {
			logInfo("Skipping execution.");
		}
	}

	private File getNpxExec() throws IOException {
		if (this.npxExec == null) {
			this.npxExec = Utils.findExecutable(Platform.CURRENT.npx, this);
		}
		return this.npxExec;
	}

	private File getWorkingDirectory() {
		if (this.workingDirectory == null) {
			// determine cds working directory if not specified
			this.workingDirectory = findCdsWorkingDir();
		} else {
			logInfo("Using configured working directory: %s", this.workingDirectory);
		}
		return this.workingDirectory;
	}

	private Semver getCdsdkVersion() throws MojoExecutionException {
		try (ByteArrayOutputStream output = new ByteArrayOutputStream()) {
			// execute cds version and catch console output
			executeCds(getWorkingDirectory(), "version", output);

			String cdsVersionOutput = new String(output.toByteArray(), StandardCharsets.UTF_8);
			logDebug(cdsVersionOutput);

			try (Scanner scanner = new Scanner(cdsVersionOutput)) {
				String line;
				while (scanner.hasNextLine()) {
					line = scanner.nextLine();
					if (line.contains("@sap/cds-dk")) {
						return new Semver(line.split(":")[1].trim());
					}
				}
			}

			throw new MojoExecutionException(format(
					"Required @sap/cds-dk not installed, version information not available:%n%s", cdsVersionOutput));

		} catch (IOException | IllegalArgumentException | IndexOutOfBoundsException e) {
			throw new MojoExecutionException(e.getMessage(), e);
		}
	}
}
