/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.maven.plugin.build;

import java.io.File;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.Component;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.settings.Proxy;
import org.apache.maven.settings.crypto.DefaultSettingsDecryptionRequest;
import org.apache.maven.settings.crypto.SettingsDecrypter;
import org.codehaus.plexus.util.StringUtils;

/**
 * Download and install a Node.js distribution on the local file system.<br>
 * If the requested version is already installed, the download and installation is skipped.<br>
 * <br>
 * This goal also recognizes proxy configurations in your <code>~/.m2/settings.xml</code>. If there’s an active proxy
 * for the host in the <code>downloadUrl</code> configured, it will be used.<br>
 * <br>
 * This goal also exposes the following locations as project properties:
 * <ul>
 * <li><strong>${cds.node.executable}</strong>: Location of the <code>node</code> executable</li>
 * <li><strong>${cds.npm.executable}</strong>: Location of the <code>npm</code> executable</li>
 * <li><strong>${cds.npx.executable}</strong>: Location of the <code>npx</code> executable</li>
 * <li><strong>${cds.node.directory}</strong>: Directory containing the Node.js executables (node/npm/npx), can be used
 * to enhance the $PATH environment variable</li>
 * </ul>
 * <br>
 *
 * @since 1.7.0
 */
@Mojo(name = "install-node", defaultPhase = LifecyclePhase.INITIALIZE, aggregator = true, requiresProject = false)
public class InstallNodeMojo extends AbstractNodejsMojo {

	@Component(role = SettingsDecrypter.class)
	private SettingsDecrypter decrypter;

	/**
	 * Where to download Node.js binary from.
	 */
	@Parameter(property = "cds.install-node.downloadUrl", defaultValue = NodeInstaller.DEFAULT_NODEJS_DOWNLOAD_ROOT, required = true)
	private String downloadUrl;

	/**
	 * Force download and installation of Node.js, even if it's already installed.
	 */
	@Parameter(property = "cds.install-node.force", defaultValue = "false")
	private boolean force = false;

	/**
	 * Location of the target installation directory. If this parameter isn’t specified, the Node.js distribution is
	 * installed into a directory within the local Maven repository:
	 * <code>~/.m2/repository/com/sap/cds/cds-maven-plugin/cache/...</code>
	 */
	@Parameter(property = "cds.install-node.installDirectory")
	private File installDirectory;

	/**
	 * The Node.js version to use. IMPORTANT! The Node.js version has to start with 'v', for example 'v12.21.0'.
	 */
	@Parameter(property = "cds.install-node.nodeVersion", defaultValue = "v12.21.0", required = true)
	private String nodeVersion;

	/**
	 * Skip execution of this goal.
	 */
	@Parameter(property = "cds.install-node.skip", defaultValue = "false")
	private boolean skip;

	@Override
	public void execute() throws MojoExecutionException {
		if (!this.skip) {
			installNode();
		} else {
			logInfo("Skipping execution.");
		}
	}

	private List<Proxy> getProxyConfig() {
		if (this.settings.getProxies() == null || this.settings.getProxies().isEmpty()) {
			return Collections.emptyList();
		}

		List<Proxy> proxies = new ArrayList<>(this.settings.getProxies().size());

		for (Proxy mavenProxy : this.settings.getProxies()) {
			if (mavenProxy.isActive()) {
				DefaultSettingsDecryptionRequest decryptionRequest = new DefaultSettingsDecryptionRequest(mavenProxy);
				mavenProxy = this.decrypter.decrypt(decryptionRequest).getProxy();
				proxies.add(mavenProxy);
			}
		}
		return proxies;
	}

	/**
	 * Download and install Node.js if requested.
	 *
	 * @throws MojoExecutionException if Node.js download or installation failed.
	 */
	private void installNode() throws MojoExecutionException {
		if (!this.nodeVersion.startsWith("v")) {
			throw new MojoExecutionException(String.format(
					"Configured Node.js version \"%s\" doesn't start with prefix 'v' as required.", this.nodeVersion));
		}
		NodeCacheResolver nodeResolver = new NodeCacheResolver(this.repositorySystemSession);

		NodeInstaller installer = new NodeInstaller(this.nodeVersion, nodeResolver, this);
		installer.setProxies(getProxyConfig());
		installer.setForce(this.force);

		if (StringUtils.isNotBlank(this.downloadUrl)) {
			installer.setDownloadRoot(this.downloadUrl);
		}

		if (this.installDirectory != null) {
			installer.setInstallDirectory(this.installDirectory);
		}
		this.installDirectory = installer.install();

		// Node.js installation succeeded and the location properties can be set
		setProperty(PROP_NODE_EXECUTABLE, installer.getNodeExec());
		setProperty(PROP_NODE_DIR, installer.getNodeExec().getParentFile());
		setProperty(PROP_NPM_EXECUTABLE, installer.getNpmExec());
		setProperty(PROP_NPX_EXECUTABLE, installer.getNpxExec());

		logDebug("node install directory: " + this.installDirectory);
	}
}
