package com.sap.cds.maven.plugin.build;

import java.io.IOException;
import java.time.Instant;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.TimeUnit;

import org.apache.maven.model.Plugin;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.shared.invoker.InvocationRequest;
import org.apache.maven.shared.invoker.InvocationResult;
import org.apache.maven.shared.utils.cli.ShutdownHookUtils;

import com.google.common.annotations.VisibleForTesting;
import com.sap.cds.maven.plugin.util.DirectoryWatcher;

/**
 * Starts a CAP Java NG application and watches for changes in the CDS model. If changes are detected, the application
 * is built and automatically restarted.
 *
 * Call <code>mvn cds:watch</code> or <code>mvn com.sap.cds:cds-maven-plugin:watch</code> to start the CAP Java NG
 * application.<br>
 *
 * <br>
 * <strong>Note:</strong> This goal can only be executed from the command line.<br>
 * <br>
 *
 * @since 1.17.0
 */
@Mojo(name = "watch", defaultPhase = LifecyclePhase.NONE, aggregator = true, requiresDirectInvocation = true)
public class WatchMojo extends AbstractInvokerMojo {

	private static final String SPRING_BOOT_MAVEN_PLUGIN = "org.springframework.boot:spring-boot-maven-plugin";

	private Process mvnProcess;

	private DirectoryWatcher watcher;

	/**
	 * A comma separated list with GLOB pattern describing directories and files to be exclude from watching for
	 * changes. Changes in these directories and files will be ignored.<br>
	 * See <a href=
	 * "https://docs.oracle.com/javase/8/docs/api/java/nio/file/FileSystem.html#getPathMatcher-java.lang.String-">here</a>
	 * for further details about the supported GLOB patterns.
	 */
	@Parameter(property = "cds.watch.excludes", defaultValue = "**/node_modules,**/target,**/gen", required = true)
	private List<String> excludes;

	/**
	 * A comma separated list of GLOB patterns describing files to be include in watching for changes. Changes on those
	 * files will trigger a build and restart of the application.<br>
	 * See <a href=
	 * "https://docs.oracle.com/javase/8/docs/api/java/nio/file/FileSystem.html#getPathMatcher-java.lang.String-">here</a>
	 * for further details about the supported GLOB patterns.
	 */
	@Parameter(property = "cds.watch.includes", defaultValue = "**/*.cds,**/*.csv", required = true)
	private List<String> includes;

	private Instant lastRestart;

	@Override
	public void execute() throws MojoExecutionException {
		ensureCliExecuted();

		ensureSpringBootPlugin();

		initialize();

		restartApplication();

		try {
			this.watcher.start();

			this.watcher.join();
		} catch (IOException e) {
			logError(e);
			throw new MojoExecutionException(e.getMessage(), e);
		} catch (InterruptedException e) {
			Thread.currentThread().interrupt();
			logError(e);
			throw new MojoExecutionException(e.getMessage(), e);
		}
	}

	@VisibleForTesting
	synchronized Instant getLastRestart() {
		return this.lastRestart;
	}

	/**
	 * Starts or restarts the spring-boot application.
	 */
	private synchronized void restartApplication() {
		this.lastRestart = Instant.now();

		// first stop a running application instance
		stopApplication();

		try {
			startApplication();
		} catch (MojoExecutionException e) {
			logError(e);
		}
	}

	private void startApplication() throws MojoExecutionException {
		InvocationRequest request = createRequest(getStartGoals());

		logInfo("Starting application %s", super.project.getName());

		this.mvnProcess = executeMaven(request);
	}

	/**
	 * Stops the spring-boot application.
	 *
	 * @return the application stop invocation {@link InvocationResult result}
	 */
	private void stopApplication() {
		if (this.mvnProcess != null) {
			logInfo("Stopping application %s", super.project.getName());

			this.mvnProcess.destroy();

			try {
				boolean exited = this.mvnProcess.waitFor(5, TimeUnit.SECONDS);
				if (exited) {
					logInfo("Stopped application %s: rc=%d", super.project.getName(), this.mvnProcess.exitValue());
				} else {
					this.mvnProcess.destroyForcibly();
					logInfo("Destroyed application %s", super.project.getName());
				}
			} catch (InterruptedException e) {
				Thread.currentThread().interrupt();
				logError(e);
			} finally {
				this.mvnProcess = null;
			}
		}
	}

	private void initialize() {
		ShutdownHookUtils.addShutDownHook(new Thread(this::stopApplication));

		this.watcher = new DirectoryWatcher(getTopmostProjectDir().toPath(), this::restartApplication, this,
				this.includes, this.excludes);
	}

	private static List<String> getStartGoals() {
		return Collections.singletonList(SPRING_BOOT_MAVEN_PLUGIN + ":run");
	}

	/**
	 * Ensures that current project uses the spring-boot-maven-plugin.
	 *
	 * @throws MojoExecutionException if project doesn't user spring-boot-maven-plugin
	 */
	private void ensureSpringBootPlugin() throws MojoExecutionException {
		Plugin plugin = this.project.getModel().getBuild().getPluginsAsMap().get(SPRING_BOOT_MAVEN_PLUGIN);
		if (plugin == null) {
			throw new MojoExecutionException("Project doesn't use " + SPRING_BOOT_MAVEN_PLUGIN
					+ " . To watch this projects this plugin required.");
		}
	}
}
