/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.maven.plugin.util;

import static java.lang.String.format;

import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStreamReader;
import java.nio.channels.FileChannel;
import java.nio.channels.FileLock;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileSystems;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.PathMatcher;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.StringTokenizer;
import java.util.function.Consumer;
import java.util.stream.Collectors;

import javax.xml.XMLConstants;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.TransformerFactoryConfigurationError;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.apache.commons.exec.CommandLine;
import org.apache.maven.project.MavenProject;
import org.w3c.dom.Document;
import org.xml.sax.SAXException;

import com.sap.cds.maven.plugin.CdsMojoLogger;

/**
 * A helper class providing utility methods.
 */
public class Utils {

	private static DocumentBuilderFactory docBuilderFactory;

	private static TransformerFactory transformerFactory;

	private Utils() {
		// to avoid instances
	}

	/**
	 * Creates an array with given arguments
	 *
	 * @param arguments the array elements
	 * @return the string array containing given arguments
	 */
	@SafeVarargs
	public static <T> T[] array(T... arguments) {
		return arguments;
	}

	/**
	 * Creates a directory including all parent directories.
	 *
	 * @param dirPath the directory path
	 * @return the created directory
	 * @throws IOException if the directory can’t be created or the file already exists but isn’t a directory
	 */
	public static File createDir(String dirPath) throws IOException {
		File directory = new File(dirPath);
		return prepareDestination(directory, true);
	}

	/**
	 * Creates a file including all parent directories.
	 *
	 * @param file the file to created
	 * @return the created file
	 * @throws IOException if the file can’t be created
	 */
	public static File createFile(File file) throws IOException {
		if (file == null || file.exists()) {
			return file;
		}
		prepareDestination(file, false);
		if (!file.createNewFile()) {
			throw new IOException(format("Failed to create file %s ", file.getAbsolutePath()));
		}
		return file;
	}

	/**
	 * @param roots         start directories for finding the directory
	 * @param directoryName the glob directory name pattern
	 * @return the file or <code>null</code> if not found
	 * @throws IOException if an I/O error occurred while finding the file
	 */
	public static File findDirectory(String directoryName, Collection<File> roots) throws IOException {
		Finder finder = new DirectoryFinder(directoryName);
		if (roots != null && !roots.isEmpty()) {
			for (File root : roots) {
				Files.walkFileTree(root.toPath(), finder);
			}
		}
		return finder.getFirstFile();
	}

	/**
	 * Finds the location of given executable on local file-system.
	 *
	 * @param execName name of executable to find
	 * @param logger   a logger instance
	 * @return a {@link File} representing the found executable or <code>null</code> if not found
	 * @throws IOException           if an I/O error occurred during search
	 * @throws FileNotFoundException if executable couldn't be found
	 */
	public static File findExecutable(String execName, CdsMojoLogger logger) throws FileNotFoundException, IOException {
		CommandLine cmdLine;
		if (Platform.CURRENT.isWindows()) {
			cmdLine = new CommandLine("where").addArgument(execName);
		} else {
			cmdLine = new CommandLine("which").addArgument(execName);
		}
		logger.logInfo("Searching %s on local file-system.", execName);
		ProcessExecutor executor = new ProcessExecutor(null, cmdLine);
		executor.setExitValues(new int[] { 0, 1 }); // avoids throwing ExecuteException if exec isn't found

		try (ByteArrayOutputStream outputStream = new ByteArrayOutputStream()) {
			int rc = executor.execute(outputStream, outputStream, logger);
			logger.logDebug("Return code of search: %d", rc);

			try (BufferedReader reader = new BufferedReader(new InputStreamReader(
					new ByteArrayInputStream(outputStream.toByteArray()), StandardCharsets.UTF_8))) {
				String location = reader.readLine();
				logger.logDebug("First location %s", location);

				while (location != null) {
					File execFile = new File(location);
					// return first available executable
					if (execFile.canExecute()) {
						logger.logInfo("Found %s", execFile.getAbsolutePath());
						return execFile;
					}
					logger.logDebug("Not an executable: %s", execFile.getAbsolutePath());

					location = reader.readLine();
					logger.logDebug("Next location %s", location);
				}
			}
		}
		throw new FileNotFoundException(format("Executable %s not found.", execName));
	}

	/**
	 * @param roots    start directories for finding the directory
	 * @param fileName the glob file name pattern
	 * @return the file or <code>null</code> if not found
	 * @throws IOException if an I/O error occurred while finding the file
	 */
	public static File findFile(String fileName, Collection<File> roots) throws IOException {
		Finder finder = new FileFinder(fileName);
		if (roots != null && !roots.isEmpty()) {
			for (File root : roots) {
				Files.walkFileTree(root.toPath(), finder);
			}
		}
		return finder.getFirstFile();
	}

	/**
	 * Returns the directory for a given Java package.
	 *
	 * @param srcBaseDir  the source base directory
	 * @param packageName the package
	 * @return the package directory
	 */
	public static File getPackageDir(File srcBaseDir, String packageName) {
		String packagePath = packageName.replace('.', File.separatorChar);
		return new File(srcBaseDir, packagePath);
	}

	/**
	 * Returns a list with all resource directories of a given Maven project.
	 *
	 * @param project the Maven project
	 * @return a list with all resource directories
	 */
	public static List<File> getResourceDirs(MavenProject project) {
		return project.getBuild().getResources().stream() //
				.map(resource -> new File(resource.getDirectory())) //
				.filter(File::exists) //
				.collect(Collectors.toList());
	}

	/**
	 * Parses a given xml file into a {@link Document}.
	 *
	 * @param file the xml file to parse
	 * @return the parsed {@link Document}
	 * @throws ParserConfigurationException if a DocumentBuilder can’t be created which satisfies the configuration
	 *                                      requested.
	 * @throws IOException                  If any IO errors occur.
	 * @throws SAXException                 If any parse errors occur.
	 */
	public static Document parse(File file) throws ParserConfigurationException, SAXException, IOException {
		DocumentBuilder documentBuilder = createDocumentBuilder();
		return documentBuilder.parse(file);
	}

	/**
	 * @param destination the destination to prepare
	 * @param directory   if given destination is a directory
	 * @return given destination file
	 * @throws IOException if preparing failed
	 */
	public static File prepareDestination(File destination, boolean directory) throws IOException {
		File dirs = directory ? destination : destination.getParentFile();
		if (!dirs.exists() && !dirs.mkdirs()) {
			throw new IOException(format("Creating directory %s failed.", directory));
		}
		return destination;
	}

	/**
	 * Runs the given {@link Runnable} if a lock could be acquired on given file.
	 *
	 * @param lockFile the {@link File} used for synchronisation
	 * @param runnable the {@link Runnable} to execute when lock could be acquired
	 * @param maxWait  max. time to wait in seconds to acquire lock
	 * @throws IOException if lock couldn't be acquired in given max. wait time
	 */
	public static void runIfLockable(File lockFile, Runnable runnable, int maxWait) throws IOException {
		Objects.requireNonNull(lockFile, "lockFile");
		Objects.requireNonNull(runnable, "runnable");

		// prepare lock file if missing
		createFile(lockFile);

		int waited = 0;
		while (waited <= (maxWait * 1000)) {
			try (FileChannel channel = FileChannel.open(lockFile.toPath(), StandardOpenOption.WRITE);
					FileLock lock = channel.tryLock()) {
				if (lock != null) {
					runnable.run();
					return;
				}
				Thread.sleep(100);
				waited += 100;
			} catch (InterruptedException e) {
				Thread.currentThread().interrupt();
				throw new IOException(e.getMessage(), e);
			}
		}
		throw new IOException(format("Failed to acquire lock within %d seconds on file %s", maxWait, lockFile));
	}

	/**
	 * Sets execution flag at given file.
	 *
	 * @param file the file
	 * @throws IOException           if setting execution flag failed
	 * @throws FileNotFoundException if file wasn't found
	 */
	public static void setExecutionFlag(File file) throws IOException, FileNotFoundException {
		if (file != null) {
			if (!file.exists()) {
				throw new FileNotFoundException(format("File %s not found.", file.getAbsoluteFile()));
			}
			if (!file.setExecutable(true)) {
				throw new IOException(format("Failed to set execution flag on %s", file.getAbsolutePath()));
			}
		}
	}

	/**
	 * Calls a given setter with given value, if value isn't <code>null</code>.
	 *
	 * @param <T>    type of value to set
	 * @param setter the setter to call
	 * @param value  the value to set
	 */
	public static <T> void setIfNotNull(Consumer<T> setter, T value) {
		if (value != null) {
			setter.accept(value);
		}
	}

	/**
	 * Splits the given string into tokens by using whitespace as delimiters.
	 *
	 * @param string the string to be split into tokens
	 * @return the tokens
	 */
	public static String[] splitByWhitespaces(String string) {
		StringTokenizer tokenizer = new StringTokenizer(string);
		String[] tokens = new String[tokenizer.countTokens()];
		for (int i = 0; i < tokens.length; i++) {
			tokens[i] = tokenizer.nextToken();
		}
		return tokens;
	}

	/**
	 * Serializes a given {@link Document} into a file.
	 *
	 * @param file the target file
	 * @param doc  the {@link Document} to store
	 * @throws TransformerFactoryConfigurationError if the XML serializer can't be created
	 * @throws TransformerException                 if serialization to XML failed
	 */
	public static void store(File file, Document doc)
			throws TransformerFactoryConfigurationError, TransformerException {
		if (transformerFactory == null) {
			transformerFactory = TransformerFactory.newInstance();
		}

		Transformer transformer = transformerFactory.newTransformer();
		transformer.setOutputProperty(OutputKeys.STANDALONE, "yes");
		transformer.setOutputProperty(OutputKeys.METHOD, "xml");
		transformer.setOutputProperty(OutputKeys.ENCODING, StandardCharsets.UTF_8.name());

		DOMSource source = new DOMSource(doc);
		StreamResult result = new StreamResult(file);
		transformer.transform(source, result);
	}

	private static DocumentBuilder createDocumentBuilder() throws ParserConfigurationException {
		if (docBuilderFactory == null) {
			docBuilderFactory = DocumentBuilderFactory.newInstance();
			docBuilderFactory.setAttribute(XMLConstants.ACCESS_EXTERNAL_DTD, "");
			docBuilderFactory.setAttribute(XMLConstants.ACCESS_EXTERNAL_SCHEMA, "");

			// fixes fortify issue: To avoid XXE injections the following properties should be set for an XML factory,
			// parser or reader
			docBuilderFactory.setFeature("http://xml.org/sax/features/external-general-entities", false);
			docBuilderFactory.setFeature("http://xml.org/sax/features/external-parameter-entities", false);

			docBuilderFactory.setNamespaceAware(true);
			docBuilderFactory.setIgnoringComments(false);
			docBuilderFactory.setIgnoringElementContentWhitespace(true);
		}
		return docBuilderFactory.newDocumentBuilder();
	}

	static class DirectoryFinder extends Finder {
		DirectoryFinder(String pattern) {
			super(pattern);
		}

		@Override
		public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) {
			find(dir);
			return FileVisitResult.CONTINUE;
		}

		@Override
		public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) {
			return FileVisitResult.CONTINUE;
		}

		@Override
		public FileVisitResult visitFileFailed(Path file, IOException exc) {
			return FileVisitResult.CONTINUE;
		}
	}

	static class FileFinder extends Finder {
		FileFinder(String pattern) {
			super(pattern);
		}

		@Override
		public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) {
			return FileVisitResult.CONTINUE;
		}

		@Override
		public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) {
			find(file);
			return FileVisitResult.CONTINUE;
		}

		@Override
		public FileVisitResult visitFileFailed(Path file, IOException exc) {
			return FileVisitResult.CONTINUE;
		}
	}

	/**
	 * A {@code FileVisitor} that finds all files that match the specified pattern.
	 */
	abstract static class Finder extends SimpleFileVisitor<Path> {
		private final List<File> files = new ArrayList<>();

		private final PathMatcher matcher;

		Finder(String pattern) {
			this.matcher = FileSystems.getDefault().getPathMatcher("glob:" + pattern);
		}

		// compares the glob pattern against the file or directory name
		void find(Path file) {
			Path name = file.getFileName();
			if (name != null && this.matcher.matches(name)) {
				this.files.add(file.toFile());
			}
		}

		File getFirstFile() {
			return !this.files.isEmpty() ? this.files.get(0) : null;
		}
	}
}
