/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.maven.plugin.add;

import java.io.File;
import java.io.IOException;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.TransformerException;

import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.project.MavenProject;
import org.codehaus.plexus.util.StringUtils;
import org.w3c.dom.Document;
import org.xml.sax.SAXException;

import com.sap.cds.maven.plugin.util.PomUtils;
import com.sap.cds.maven.plugin.util.Utils;

/**
 * Add a target platform <code>cds-starter-${targetPlatform}</code> to the dependencies of the current CAP Java project
 * if missing.<br>
 * Call <code>mvn com.sap.cds:cds-maven-plugin:addTargetPlatform -DtargetPlatform=cloudfoundry</code> to add Cloud
 * Foundry target platform.<br>
 * <br>
 * <strong>Note:</strong> This goal can only be executed from the command line.<br>
 * <br>
 *
 * @since 1.5.0
 */
@Mojo(name = "addTargetPlatform", aggregator = true, requiresDirectInvocation = true)
public class AddTargetPlatformMojo extends AbstractAddMojo {

	private boolean added;

	/**
	 * The target platform support to add.
	 */
	@Parameter(property = "targetPlatform", required = true)
	private String targetPlatform;

	@Override
	public void execute() throws MojoExecutionException, MojoFailureException {
		ensureCliExecuted();

		MavenProject srvProject = findSrvProject();

		updatePom(srvProject.getFile());
	}

	/**
	 * Just for testing purpose.
	 *
	 * @return <code>true</code> if dependency was added
	 */
	boolean isAdded() {
		return this.added;
	}

	private void updatePom(File pomFile) throws MojoExecutionException {
		if (StringUtils.isEmpty(this.targetPlatform)) {
			throw new MojoExecutionException("Target platform not specified.");
		}
		this.targetPlatform = this.targetPlatform.trim();

		String artifactId = null;
		if (this.targetPlatform.equalsIgnoreCase("cloudfoundry") || this.targetPlatform.equalsIgnoreCase("cf")) {
			artifactId = "cds-starter-cloudfoundry";
		}

		if (artifactId == null) {
			throw new MojoExecutionException("Unknown target platform: " + targetPlatform);
		}

		try {
			Document doc = Utils.parse(pomFile);

			this.added = PomUtils.addDependency(doc, "com.sap.cds", artifactId, null);
			if (this.added) {
				logInfo("Added dependency: %s", artifactId);
				Utils.store(pomFile, doc);
			} else {
				logWarn("Dependency %s already present, nothing added.", artifactId);
			}
		} catch (ParserConfigurationException | SAXException | IOException | TransformerException e) {
			throw new MojoExecutionException(e.getMessage(), e);
		}
	}
}
